/*
 * This file is part of PARAM.
 *
 * PARAM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PARAM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with PARAM. If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2009-2010 Ernst Moritz Hahn (emh@cs.uni-sb.de)
 */

#ifndef CONTROLLER_H
#define CONTROLLER_H

#include <vector>
#include <list>
#include <iosfwd>
#include <boost/program_options.hpp>
#include "PMC.h"
#include "prismparser/Model.h"
#include "rationalFunction/RationalFunction.h"
#include "rationalFunction/Base.h"
#include "rationalFunction/CVC3Converter.h"

namespace prismparser {
  class Model;
}

namespace parametric {
  enum PartRefOrder {
    bigFirst,
    smallFirst,
    firstFirst,
    lastFirst
  };

  class RegionScheduler;
  class Statistics;
  
  typedef std::set<PMM::state> StateSet;
  typedef std::list<PMM::state> StateList;
  typedef std::vector<PMM::state> StateVector;
  typedef std::map<PMM::state, Rational::RationalFunction> RewardMap;
  typedef std::map<PMM::state, Rational::RationalFunction> StateMap;
  typedef HashMap<PMM::state,Rational::RationalFunction> Results;
  
  class PASSStateExplorer;
  class Quotient;
  class Refiner;
  class WeakRefiner;
  class StrongRefiner;
  class StateProp;
  class Gauss;
  class SteadyState;
  
  class Controller {
    /* too many friends, something like package visibility would be
     * more appropriate here.
     */
    friend class PASSStateExplorer;
    friend class LowLevelStateExplorer;
    friend class Quotient;
    friend class Refiner;
    friend class WeakRefiner;
    friend class StrongRefiner;
    friend class StateProp;
    friend class Gauss;
    friend class SteadyState;
    friend class MayChange;
  public:
    /**
     * Type for specifying the kind of analysis to be done
     */
    enum AnalysisType {
      unboundedUntilAnalysis,
      boundedUntilAnalysis,
      reachabilityRewardAnalysis,
    };
    
    void setAnalysisType(AnalysisType);
    bool needsStrongBisimulation() const;
    bool isRewardAnalysis() const;
    bool isDiscreteTime() const;
    Controller(int, char **);
    ~Controller();
    void execute(RegionScheduler &);
    void embed();
    void unboundedUntil(RegionScheduler &);
    void boundedUntil(RegionScheduler &);
    void printDOT(const std::string &) const;
    void printDOT(std::ostream &) const;
    void printDOT() const;
    void parse();
    void createSimpleRegionScheduler(const Results &, RegionScheduler &);
    
    PMC &getPMC();
    void printStatistics();
    const StateSet &getInitStates() const;
    const StateSet &getTargetStates() const;

    double time;
  private:
    void getParamsFromModel(prismparser::Model &);
    void prepareSymbols(prismparser::Model &);
    void printStartMessage() const;
    
    Statistics *statistics;
    Rational::CVC3Converter cvc3Converter;
    AnalysisType analysisType;
    prismparser::ModelType model_type;
    boost::program_options::variables_map vm;
    std::vector<std::string> nondetParams;
    PMC *pmc;
    StateSet initStates;
    StateSet targetStates;
    RewardMap stateRewards;
  };
}

#endif
