/*
 * This file is part of PARAM.
 *
 * PARAM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PARAM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with PARAM.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2010 Ernst Moritz Hahn (emh@cs.uni-sb.de)
 */

#include <string>
#include <iostream>
#include "ProgramOptions.h"

namespace parametric {
  
  using namespace std;
  namespace po = boost::program_options;
  
  /**
   * Show help on program usage.
   *
   * @param calledName name under which PARAM was called
   * @param visible named program options visible to user
   */
  void showHelp(const char *calledName,
                const boost::program_options::options_description &visible) {
    cout << "Usage: " << calledName
         << " <model-file> <properties-file [options]" << endl;
    cout << visible << endl;
  }
  
  
  /**
   * Create options map from command line.
   *
   * @param argc number of command line parameters
   * @param argv command line parameters
   * @return options read from command line
   */
  po::variables_map parseCommandLine(int argc, char *argv[]) {
    /* named options */
    po::options_description bisimulation("Bisimulation");
    bisimulation.add_options()
      ("nolump", "disable lumping")
      ("lump-method", po::value<string>()->default_value("auto"),
       "lumping method ([auto], weak, strong)")
      ("refine-order", po::value<string>()->default_value("small-first"),
       "Order of equivalence class refinement "
       "([small-first], big-first, first-first, last-first")
      ;
    
    po::options_description stateElimination("State-elimination");
    stateElimination.add_options()
      ("elimination-order", po::value<string>()->default_value("backward"),
       "order of state elimination ([backward], forward, random)")
      ;
    
    po::options_description output("Output");
    output.add_options()
      ("result-file", po::value<string>()->default_value("result"),
       "Filename prefix to output result to")
      ("result-format", po::value<string>()->default_value("regions"),
       "Output format ([regions], gnuplot, dat)")
      ("plot-step", po::value<string>()->default_value("1/100"),
       "Relative plot step width")
      ("model-dot", po::value<string>(),
       "Filename to output GraphViz graph of model")
      ("lumped-dot", po::value<string>(),
       "Filename to print lumped model as GraphViz")
      ("statistics-file", po::value<string>(),
       "Filename to output statistics")
      ;

    po::options_description visible("Options");
    visible.add_options()
      ("help", "produce help message")
      ("low-level-input", "use low-level input format")
      ;
    
    visible.add(bisimulation);
    visible.add(stateElimination);
    visible.add(output);
    
    /* the two positional values model and property file */
    po::positional_options_description pos;
    po::options_description hidden("Hidden options");
    hidden.add_options()
      ("model-file", po::value<string>(), "model to use")
      ("formula-file", po::value<string>(), "formula file to use")
      ;
    
    po::options_description cmdline_options;
    cmdline_options.add(visible).add(hidden);
    
    pos.add("model-file", 1);
    pos.add("formula-file", 2);;
    
    /* parse options */
    po::variables_map vm;
    try {
      po::store(po::command_line_parser(argc, argv).
                options(cmdline_options).positional(pos).run(), vm);
      po::notify(vm);
    } catch (...) {
      throw runtime_error("Error on parsing command line");
    }
    
    if (vm.count("help")
        || (0 == vm.count("model-file"))) {
      showHelp(argv[0], visible);
    }
    
    return vm;
  }
}
