#ifndef SPARSE_H
#define SPARSE_H

#include <vector>
#include <iosfwd>
#include <string>

namespace prohver {
  class Sparse {
  public:
    typedef std::vector<double> NonZeros;
    typedef std::vector<double> TransRewards;

    inline double calcRateSum(unsigned state) {
      unsigned stateBegin = rows[state];
      unsigned stateEnd = rows[state+1];
      double rate = 0.0;
      for (unsigned trans = stateBegin; trans < stateEnd; trans++) {
	rate += nonZeros[trans];
      }
      return rate;
    }

    /**
     * Computes the maximal outgoing rate of all states of the model.
     *
     * @return maximal outgoing rate of all states
     */
    inline double calcMaxRateSum() {
      double result = 0.0;
      for (unsigned state = 0; state < numStates; state++) {
	result = std::max(result, calcRateSum(state));
      }
      return result;
    }

    inline void addState() {
      numStates++;
    }

    inline void finishState() {
      rows.push_back(cols.size());
    }

    inline const std::vector<unsigned> &getInit() const {
      return init;
    }

    inline unsigned getNumStates() const {
      return numStates;
    }

    inline unsigned getNumFinishedStates() const {
      return rows.size() - 1u;
    }

    inline unsigned getNumTransitions() const {
      return nonZeros.size();
    }

    inline const std::vector<double> &getNonZeros() const {
      return nonZeros;
    }

    inline unsigned getTarget(unsigned trans) const {
      return cols[trans];
    }

    inline unsigned getStateBegin(unsigned state) const {
      return rows[state];
    }

    inline unsigned getStateEnd(unsigned state) const {
      return rows[state + 1];
    }

    inline const std::vector<double> &getTransRewards() const {
      return transRewards;
    }

    inline const std::vector<unsigned> &getRows() const {
      return rows;
    }

    inline const std::vector<unsigned> &getCols() const {
      return cols;
    }

    inline Sparse() {
      numStates = 0;
      vertices_added = 0;
      rows.push_back(0);
    }

    inline void addTransition(unsigned from, unsigned to, double rate) {
      cols.push_back(to);
      nonZeros.push_back(rate);
    }

    inline void addTransition
      (unsigned from, unsigned to, double rate, double reward) {
      cols.push_back(to);
      nonZeros.push_back(rate);
      transRewards.push_back(reward);
    }

    inline void close() {
      while (rows.size() < numStates + 1) {
	rows.push_back(cols.size());
	vertices_added++;
      }
    }

    inline void open() {
      for (unsigned i = 0; i < vertices_added; i++) {
	rows.pop_back();
      }
      vertices_added = 0;
    }

    inline void setInit(const std::vector<unsigned> &__init) {
      init = __init;
    }

    void print();
    void print(std::ostream &);
    void printMRMCTra();
    void printMRMCTra(const std::string &);
    void printMRMCTra(std::ostream &);
    void printMRMCLab();
    void printMRMCLab(const std::string &);
    void printMRMCLab(std::ostream &);
    void printMRMCRew(std::vector<double> &);
    void printMRMCRew(const std::string &, std::vector<double> &);
    void printMRMCRew(std::ostream &, std::vector<double> &);
    void clear();

    unsigned numStates;
    unsigned vertices_added;
    std::vector<double> nonZeros;
    std::vector<double> transRewards;
    std::vector<unsigned> cols;
    std::vector<unsigned> rows;
    std::vector<unsigned> init;
    std::vector<double> transReward;
  };
}

#endif
