#include <stdlib.h>
#include <sys/types.h>
#include <math.h>
#include <float.h>
#include <fstream>
#include <iostream>
#include <iomanip>
#include <boost/program_options.hpp>

#include "SparseLoader.h"

namespace prohver {
  using namespace std;

  /**
   * Reads a line from a label to distributions file.
   * Lines are of scheme "<label> <distribution> <prob>" with types
   * "%s %d %lf".
   *
   * @param file file to read from
   * @param label will contain label name of line
   * @param distr will contain distribution number of line
   * @param prob will contain probability of line
   * @return true iff succeeded
   */
  bool SparseLoader::readLabel2DistrLine
  (char *line, string &label, unsigned &distr, double &prob) {
    istringstream lineStream(line);
    if (!(lineStream >> label)) {
      return false;
    }
    if (!(lineStream >> distr)) {
      cerr << "==> " << string(line) << "\n";
      cerr << "Invalid file format" << endl;
      exit(EXIT_FAILURE);
    }
    if (!(lineStream >> prob)) {
      cerr << "Invalid file format" << endl;
      exit(EXIT_FAILURE);
    }
    return true;
  }

  /**
   * Load DISTRIBUTIONS section
   * consisting of multiple lines of the scheme "<label>
   * <distr> <prob>" with types "%d %d %lf".
   *
   * @param line buffer line
   */
  void SparseLoader::readDistributions(char line[LINE_MAX]) {
    string labelName;
    unsigned distr;
    double prob;
    unsigned number = 0;
    unsigned lastDistr = 0;

    graphStream.getline(line, LINE_MAX);
    while (strcmp("TRANSITIONS", line) != 0) {
      readLabel2DistrLine(line, labelName, distr, prob);
      if (lastDistr != distr) {
        distrSizes.push_back(number);
        number = 0;
        lastDistr = distr;
      }
      name2label[labelName] = label2Distr.size();
      label2Distr.push_back(distr);
      label2Prob.push_back(prob);
      label2Number.push_back(number);
      number++;
      graphStream.getline(line, LINE_MAX);
    }

    distrSizes.push_back(number);
  }

  void SparseLoader::buildChoice
  (unsigned choice, unsigned source, unsigned slotNr,
   vector<unsigned> &targets, vector<double> &probs) {
    if (targetSlots[choice].size() == slotNr) {
      if (0 != targets.size()) {
        for (unsigned targetNr = 0; targetNr < targets.size(); targetNr++) {
          unsigned target = targets[targetNr];
          double prob = probs[targetNr];
          sparse.addTransition(source, target, prob);
        }
        sparse.finishChoice();
      }
    } else {
      const vector<unsigned> &stateSlot(targetSlots[choice][slotNr]);
      const vector<double> &probSlot(probSlots[choice][slotNr]);
      if (stateSlot.size() > 0) {
        for (unsigned stateNr = 0; stateNr < stateSlot.size(); stateNr++) {
          unsigned state = stateSlot[stateNr];
          double prob = probSlot[stateNr];
          targets.push_back(state);
          probs.push_back(prob);
          buildChoice(choice, source, slotNr + 1, targets, probs);
          targets.pop_back();
          probs.pop_back();
        }
      } else {
        buildChoice(choice, source, slotNr + 1, targets, probs);        
      }
    }
  }

  void SparseLoader::buildChoice(unsigned choice, unsigned source) {
    vector<unsigned> targets;
    vector<double> probs;
    buildChoice(choice, source, 0, targets, probs);
  }

  /**
   * Convert out labels read from graph to set of distributions.
   *
   * @param source source state to add set to
   */
  void SparseLoader::buildChoices(unsigned source) {
    for (unsigned choice = 0; choice < distrSizes.size(); choice++) {
      vector<unsigned> targets;
      vector<double> probs;
      buildChoice(choice, source, 0, targets, probs);
    }
  }

  /**
   * Prepare distribution slot @a distr for new source state.
   *
   * @a distr distribution slot to prepare for new source state
   */
  void SparseLoader::prepareDistrSlots(unsigned distr) {
    unsigned distrSize = distrSizes[distr];
    targetSlots[distr].resize(distrSize);
    probSlots[distr].resize(distrSize);
    for (unsigned number = 0; number < targetSlots[distr].size(); number++) {
      targetSlots[distr][number].clear();
      probSlots[distr][number].clear();
    }
  }

  /**
   * Prepare all distribution slots for new source state.
   */
  void SparseLoader::prepareDistrSlots() {
    targetSlots.resize(distrSizes.size());
    probSlots.resize(distrSizes.size());
    for (unsigned distr = 0; distr < distrSizes.size(); distr++) {
      prepareDistrSlots(distr);
    }
  }

  bool SparseLoader::readTransitionLine
  (char line[LINE_MAX], unsigned &source, unsigned &target, unsigned &label) {
    char labelName[LINE_MAX];
    map<string, unsigned>::iterator it;

    numTransitions++;
    sscanf(line, "%u %u %s", &source, &target, labelName);
    it = name2label.find(string(labelName));

    if (it != name2label.end()) {
      label = it->second;
    }
    else {
      cerr << "Unknown label name \"" << labelName << "\"" << endl;
      exit(EXIT_FAILURE);
    }
    return true;
  }

  /**
   * Read set of initial states from input stream.
   *
   * @param line buffer line
   */
  void SparseLoader::readInitStates(char line[LINE_MAX]) {
    inits.clear();
    graphStream.getline(line, LINE_MAX);
    while (strcmp("UNSAFE", line) != 0) {
      unsigned init = atoi(line);
      inits.push_back(init);
      graphStream.getline(line, LINE_MAX);
    }
  }

  /**
   * Read set of unsafe states from input stream.
   *
   * @param line buffer line
   */
  void SparseLoader::readUnsafeStates(char line[LINE_MAX]) {
    targets.clear();
    graphStream.getline(line, LINE_MAX);
    while (strcmp("DISTRIBUTIONS", line) != 0) {
      unsigned target = atoi(line);
      targets.push_back(target);
      graphStream.getline(line, LINE_MAX);
    }
  }

  /**
   * Read transitions between states.
   *
   * @param line buffer line
   */
  void SparseLoader::readTransitions(char line[LINE_MAX]) {
    unsigned lastSource = 0;
    unsigned numStates = 1;
    sparse.addState();
    unsigned source = 0;
    graphStream.getline(line, LINE_MAX);
    unsigned target;
    unsigned label;
    readTransitionLine(line, source, target, label);
    prepareDistrSlots();
    numTransitions = 0;

    while ((strcmp("END", line) != 0)
           && !graphStream.eof()) {
      readTransitionLine(line, source, target, label);
      numStates = max(numStates, source + 1);
      numStates = max(numStates, target + 1);

      unsigned distr = label2Distr[label];
      /* if we have new source state or distribution, finish old one */
      if (source > lastSource) {
        buildChoices(lastSource);
        prepareDistrSlots();
        while (lastSource != source) {
          sparse.finishState();
          sparse.addState();
          lastSource++;
        }
      }
      unsigned number = label2Number[label];
      double prob = label2Prob[label];
      targetSlots[distr][number].push_back(target);
      probSlots[distr][number].push_back(prob);
      graphStream.getline(line, LINE_MAX);
    }
    buildChoices(source);
    sparse.finishState();
    while (sparse.getNumStates() < numStates) {
      sparse.addState();
      sparse.finishState();
    }
    sparse.close();
  }

  /**
   * Add set of initial states to MDP.
   */
  void SparseLoader::addInits() {
    for (unsigned initNr = 0; initNr < inits.size(); initNr++) {
      unsigned init = inits[initNr];
      sparse.addInit(init);
    }
  }
  
  /**
   * Add set of unsafe states to iteration vectors.
   */
  void SparseLoader::addUnsafes() {
    for (unsigned targetNr = 0; targetNr < targets.size(); targetNr++) {
      unsigned target = targets[targetNr];
      iters.targets[target] = true;
    }
  }

  /**
   * Clear all helper structures to create MDP.
   */
  void SparseLoader::clearAll() {
    for (unsigned targetNr = 0; targetNr < targetSlots.size(); targetNr++) {
      for (unsigned number = 0; number < targetSlots[targetNr].size();
           number++) {
        targetSlots[targetNr][number].clear();
        probSlots[targetNr][number].clear();
      }
    }
    iters.resize(0);
    sparse.clear();
    inits.clear();
    targets.clear();
  }

  /**
   * Read next transition system and create MDP and iteration vectors.
   *
   * @return true iff transition system could be read
   */
  bool SparseLoader::readNext() {
    bool foundInit(false);
    bool foundUnsafe(false);
    bool foundTransitions(false);
    bool foundDistributions(false);
    clearAll();
    bool done = false;
    char line[LINE_MAX];
    while (!done && !graphStream.eof()) {
      graphStream.getline(line, LINE_MAX);
      if (strcmp("INIT", line) == 0) {
        readInitStates(line);
        foundInit = true;
      } 
      if (strcmp("UNSAFE", line) == 0) {
        readUnsafeStates(line);
        foundUnsafe = true;
      }
      if (strcmp("DISTRIBUTIONS", line) == 0) {
	readDistributions(line);
	foundDistributions = true;
      }
      if (strcmp("TRANSITIONS", line) == 0) {
        readTransitions(line);
        foundTransitions = true;
      }
      if (strcmp("END", line) == 0) {
        done = true;
      }
    }
    if (foundInit && foundUnsafe && foundDistributions && foundTransitions) {
      iters.resize(sparse.getNumStates());
      addInits();
      addUnsafes();
      return true;
    } else {
      return false;
    }
  }

  /**
   * Read sparse MDP as produced by HSolver
   * @a sparse must be an empty MDP sparse matrix
   * @todo describe file format
   * @param filename filename to read model from
   * @param will contain MDP afterwards
   */
  SparseLoader::SparseLoader(istream &graphStream__,
                             IterationVectorsMDP &iters__)
    : iters(iters__),
      sparse(iters__.getModel()),
      graphStream(graphStream__) {
  }

}
