/***************************************************************************
 *   Copyright (C) 2005 by Goran Frehse   *
 *   goran.frehse@imag.fr   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "continuous_set.h"
 
using namespace std;

   // Constructors
   continuous_set*
   continuous_set::clone() const
   {
      continuous_set* pcs=clone_noids();
      pcs->set_pmy_variable_id_map(pmy_variable_id_map);
   }
   
   continuous_set* 
   continuous_set::construct_universe( dimension_t d, const variable_id_map_ptr pnew_map )
   {
      continuous_set* pcs=construct_universe_noids(d);
      if (pnew_map)
         pcs->set_pmy_variable_id_map(pnew_map);
   }
   
   continuous_set* 
   continuous_set::construct_empty( dimension_t d, const variable_id_map_ptr pnew_map )
   {
      continuous_set* pcs=construct_empty_noids(d);
      if (pnew_map)
         pcs->set_pmy_variable_id_map(pnew_map);
   }
   

   // Non-modifying non-semantic methods
   int 
   continuous_set::get_memory() const 
   {
      return 0;
   }

// Non-modifying semantic methods

//	dimension_t get_dimension() const = 0;

   dimension_t continuous_set::get_occupied_dimension() const 
   {
      return get_dimension();
   };

// bool is_empty() const = 0;
   
   bool 
   continuous_set::is_universe() const // default implementation : complement is empty
   { 
      continuous_set* pc=get_complement();
      bool b=pc->is_empty();
      delete pc;
      return b;
   }
   
   bool 
   continuous_set::is_disjoint_from(const continuous_set* pcs) const // default implementation : intersection is empty
   {
      continuous_set* pc=get_intersection(pcs);
      bool b=pc->is_empty();
      delete pc;
      return b;   
   }
   
   bool 
   continuous_set::contains(const continuous_set* pcs) const // default implementation : difference assign and emptiness
   {
      continuous_set* pc=get_difference(pcs);
      bool b=pc->is_empty();
      delete pc;
      return b;      
   }
   
	// Linear programming
//	bool minimize(const Linear_Expression &expr, Integer &inf_n, Integer &inf_d, bool &minimum) const = 0;
//   bool maximize(const Linear_Expression &expr, Integer &sup_n, Integer &sup_d, bool &maximum) const; // default implementation: minimize (-expr,...)
  
   // Methods for modifying dimensions
   void 
   continuous_set::map_space_dimensions( const PFunction& pfunc, const variable_id_map_ptr pnew_map) // pnew_map provides information in case new ids occur in the map
   {
      map_space_dimensions_noids(pfunc);
      if (pnew_map)
      {
         assert(get_dimension()<=pnew_map->size());
      }
      set_pmy_variable_id_map(pnew_map);
   }
      

   
   void 
   continuous_set::dimension_move_assign(const dimension_t& x1, const dimension_t& x2, const dimension_t& y, variable_id_map_ptr pnew_map ) // move vars x1--x2 to start at y
   {
      PFunction pfunc;
      pfunc.move_assign(x1,x2,y,get_dimension());
      if (pnew_map)
         map_space_dimensions(pfunc,pnew_map);      
   }
   
   void 
   continuous_set::add_space_dimensions_and_embed( dimension_t m, const variable_id_map_ptr pnew_map )
   {
      add_space_dimensions_and_embed_noids(m);
      // fix variable_names
      if (pnew_map)
      {
         assert(get_dimension()<=pnew_map->get_dimension());
      }
      set_pmy_variable_id_map(pnew_map);      
   }
   
  	void 
   continuous_set::add_space_dimensions_and_embed_before( dimension_t m, const variable_id_map_ptr pnew_map )
   {
      // add and then move up
      if (m>0)
      {
         dimension_t n=get_dimension();
         add_space_dimensions_and_embed(m,pnew_map);
         dimension_move_assign(0,n-1,m,pnew_map);
         // fix variable_names
         if (pnew_map)
         {
            assert(get_dimension()<=pnew_map->get_dimension());
         }
         set_pmy_variable_id_map(pnew_map);
      }
   }
   
   void 
   continuous_set::add_space_dimensions_and_embed_double( dimension_t m, const variable_id_map_ptr pnew_map )
   {
		// for sets of the form (x_0, ..., x_(n-1),y_0, ..., y_(n-1))
		// Note: This implies that the number of dimensions is even
		
      if (m>0)
		{
         if (get_dimension() % 2 == 0)
         {
            dimension_t olddim=get_dimension();
            add_space_dimensions_and_embed(2*m,pnew_map);
            if (olddim>0)
               dimension_move_assign(olddim/2,olddim-1,(olddim/2)+m,pnew_map);
            // fix variable_names
            if (pnew_map)
            {
               assert(get_dimension()<=pnew_map->get_dimension());
            }
            set_pmy_variable_id_map(pnew_map);      
         }
         else
            throw_error("Odd number of dimensions in add_space_dimensions_and_embed_double");
      }
   }
      
   void 
   continuous_set::add_space_dimensions_and_project( dimension_t m, const variable_id_map_ptr pnew_map ) // new dimensions are initialized to zero
   {
      add_space_dimensions_and_project_noids(m);
      // fix variable_names
      if (pnew_map)
      {
         assert(get_dimension()<=pnew_map->get_dimension());
      }
      set_pmy_variable_id_map(pnew_map);
   }
   
   void 
   continuous_set::remove_space_dimensions(const variable_ref_set& to_be_removed)
   {
      remove_space_dimensions_noids(to_be_removed);
      
      if (pmy_variable_id_map)
      {
         pmy_variable_id_map->erase_ids(to_be_removed);
      }
   }
   
   void 
   continuous_set::remove_space_dimensions(const dimension_t& x1, const dimension_t& x2)
   {
      variable_ref_set vs;
      for (dimension_t i=x1; i<=x2; ++i)
      {
         vs.insert(i);
      }
      remove_space_dimensions(vs);   
   }
   
   void 
   continuous_set::remove_space_dimensions(const variable_name_set& to_be_removed)
   {
      // add the variables to to_be_removed by names
      variable_ref_set vs;
      for (variable_name_set::const_iterator it=to_be_removed.begin(); it!=to_be_removed.end(); ++it)
      {
         vs.insert(pmy_variable_id_map->get_id(*it));
      };
      remove_space_dimensions(vs);         
   }

   void 
   continuous_set::concatenate_assign(const continuous_set* pcs) // produce P x Q, where P=[this] and Q=cvs
   {
      // get common vim
      variable_id_map_ptr pnvim=variable_id_map_ptr();
      if (pmy_variable_id_map && pcs->pmy_variable_id_map) // only remap if both have a map
      {
//        pnvim=pmy_variable_id_map->get_unified_map(pcs->pmy_variable_id_map);
         pmy_variable_id_map->union_assign(*pcs->pmy_variable_id_map);
      }
      dimension_t d=get_dimension();
      add_space_dimensions_and_embed(pcs->get_dimension(),pnvim);
      continuous_set* pcs2=pcs->clone();
      pcs2->add_space_dimensions_and_embed_before(d,pnvim);
      intersection_assign(pcs2);
      delete pcs2;
   }

   // Modifying methods with 0 continuous_set argument
   void 
   continuous_set::assign_variable_names( const variable_id_map_ptr pnew_map )
   {
      // to do: consistency check
//      pmy_variable_id_map->is_consistent_with(pnew_map); // maps all the ids occuring in the cyrrent map
      set_pmy_variable_id_map(pnew_map);
   }
   
   //continuous_set* get_convex_hull() = 0;
   //void simplify() = 0;
   
   // Modifying methods with 1 continuous_set argument
	void 
   continuous_set::swap(continuous_set& cvs)
   {
      swap_noids(cvs);
      variable_id_map_ptr p=pmy_variable_id_map;
      pmy_variable_id_map=cvs.pmy_variable_id_map;
      cvs.pmy_variable_id_map=p;
   }
   
   //void time_elapse_assign(const continuous_set* tp) = 0;
   bool 
   continuous_set::operator==(const continuous_set& cs) const // call is_equal(this,pcs), which has default based on mutual containment
   {  
      return contains(&cs) && cs.contains(this);
   }
   
   continuous_set* 
   continuous_set::get_intersection(const continuous_set* p1) const // caller function for get_intersection(this,p2)
   {
      return get_intersection_impl(this,p1);
   }
   
   void 
   continuous_set::intersection_assign(const continuous_set* p1) // caller function for this = get_intersection(this,p2)
   {
      continuous_set* pres=get_intersection_impl(this,p1);
      swap(*pres);
      delete pres;
   }
   
   continuous_set* 
   continuous_set::get_union(const continuous_set* p1) const // caller function for get_intersection(this,p2)
   {
      return get_union_impl(this,p1);
   }
   
   void 
   continuous_set::union_assign(const continuous_set* p1) // caller function for this = get_intersection(this,p2)
   {
      continuous_set* pres=get_union_impl(this,p1);
      swap(*pres);
      delete pres;
   }
   
   continuous_set* 
   continuous_set::get_difference(const continuous_set* p1) const // caller function for get_intersection(this,p2)
   {
      return get_difference_impl(this,p1);
   }
   
   void 
   continuous_set::difference_assign(const continuous_set* p1) // caller function for this = get_intersection(this,p2)
   {
      continuous_set* pres=get_difference_impl(this,p1);
      swap(*pres);
      delete pres;
   }

   void 
   continuous_set::add_space_dimensions_and_embed_noids( dimension_t m )
   {
      if (m>0)
      {
         PFunction pfunc;
         dimension_t n=get_dimension();
         if (n>0)
            --n;
         pfunc.move_assign(0,n,get_dimension(),get_dimension());
         map_space_dimensions(pfunc,variable_id_map_ptr());      
      }
   }
         
// Operators that possibly take as arguments two continuous_sets of different derived classes

   continuous_set* get_union_impl(const continuous_set* p1, const continuous_set* p2)
   {
      // caller function for all combinations of types
   }
   continuous_set* get_intersection_impl(const continuous_set* p1, const continuous_set* p2)
   {
   }
   
   continuous_set* get_difference_impl(const continuous_set* p1, const continuous_set* p2)
   {
   }
   
   void 
   continuous_set::set_pmy_variable_id_map( const variable_id_map_ptr pnew_map ) // no checks for consistency   
   {
      pmy_variable_id_map=pnew_map;
   }
