#include <stdlib.h>
#include <sys/types.h>
#include <math.h>
#include <float.h>
#include <fstream>
#include <iostream>
#include <iomanip>
#include <boost/program_options.hpp>

#include "Controller.h"
#include "SparseNonDet.h"
#include "SparseLoader.h"
#include "MDPExporter.h"
#include "DOTExporter.h"
#include "PARAMExporter.h"
#include "IterationVectorsMDP.h"

using namespace std;
using namespace boost;
namespace po = boost::program_options;

extern int line_number;       // current line number
extern std::string file_name; // name of the file currently being parsed

namespace prohver {

  /**
   * Prints result of one probabilistic analysis run.
   * Will print maximal reachability from an inititial state to
   * unsafe states. If "print-all-probs" is given as a command
   * line parameter, will additionally print maximal reachability
   * probabilities for each state of the abstract model. If
   * "graph-output" is given, will additionally print the MDP
   * to the given file.
   */
  void Controller::printResults
  (SparseLoader &loader, IterationVectorsMDP &iters) {
    const SparseNonDet &sparse(iters.getModel());
    const vector<unsigned> &init(sparse.getInit());
    double maxVal = 0.0;
    for (unsigned stateNr = 0; stateNr < init.size(); stateNr++) {
      unsigned state = init[stateNr];
      maxVal = max(maxVal, iters.result[state]);
    }

    if (vm.count("print-all-probs")) {
      for (unsigned state = 0; state < sparse.getNumStates(); state++) {
        cout << "VAL[" << state << "] = " << iters.result[state] << endl;
      }
    }

    cout << maxVal << " " << loader.getNumStates() << " "
         << loader.getNumTransitions() << endl;
    printGraph(iters);
  }

  /**
   * Exports MDP read into user-readable graph format.
   *
   * @param iters iteration vectors (includes MDP)
   */
  void Controller::printGraph
  (IterationVectorsMDP &iters) {
    if (vm.count("graph-output")) {
      MDPExporter *exporter;
      if ("dot" == vm["graph-output-format"].as<string>()) {
        exporter = new DOTExporter();
      } else if ("param" == vm["graph-output-format"].as<string>()) {
	exporter = new PARAMExporter();
      } else if ("yed" == vm["graph-output-format"].as<string>()) {
	//        exporter.reset(new YEdExporter());
      } else if ("aisee" == vm["graph-output-format"].as<string>()) {
	//        exporter.reset(new AiSeeExporter());
      }

      exporter->setOutput(vm["graph-output"].as<string>());
      exporter->setIterationVectors(iters);
      exporter->setPrintMarkovChain(vm["print-markov-chain"].as<bool>());
      exporter->exportGraph();
      delete exporter;
    }
  }
    
  /**
   * Construct new controller with given options.
   *
   * @param argc number of command line arguments
   * @param argv array of command line arguments
   */
  Controller::Controller(int argc, char *argv[]) {
    readOptions(argc, argv);
    SparseNonDet sparse;
    IterationVectorsMDP iters(sparse);
    iters.setPrecision(precision);
    iters.setMinProb(minProb);

    istream graphStream(cin.rdbuf());
    ifstream file;
    if (1 == vm.count("graph-input")) {
      file.open(vm["graph-input"].as<string>().c_str(), ios::in);
      graphStream.rdbuf(file.rdbuf());
    }

    SparseLoader loader(graphStream, iters);
    while (!graphStream.eof()) {
      if (loader.readNext()) {
        iters.unboundedReachBackwards();
        printResults(loader, iters);
      }
    }
  }

  /**
   * Reads options from command line
   *
   * @param argc number of command line arguments
   * @param argv array of command line arguments
   */
  void Controller::readOptions(int argc, char *argv[]) {
    parseCommandLine(argc, argv, vm);
    
    verbose = vm.count("verbose");
    precision = vm["precision"].as<double>();
    minProb = vm["min-prob"].as<bool>();
   }

  /**
   * Show help on program usage.
   *
   * @param calledName name under which PARAM was called
   * @param visible named program options visible to user
   */
  void Controller::showHelp(const char *calledName,
			    const po::options_description &visible) {
    cout << "Usage: " << calledName << endl;
    cout << visible << endl;
  }

  /**
   * Create options map from command line.
   *
   * @param argc number of command line parameters
   * @param argv command line parameters
   * @return options read from command line
   */
  void Controller::parseCommandLine(int argc, char *argv[],
				    po::variables_map &vm) {
    po::options_description visible("Options");
    visible.add_options()
      ("help", "produce help message")
      ("verbose", "verbose mode")
      ("print-all-probs", "print probabilities for all states")
      ("precision", po::value<double>()->default_value(1.0E-12),
       "Analysis precision")
      ("graph-output-format", po::value<string>()->default_value("dot"),
       "file for graph output")
      ("graph-input", po::value<string>(), "graph file to read")
      ("graph-output", po::value<string>(),
       "file for graph output")
      ("print-markov-chain", po::value<bool>()->default_value(true),
       "print underlying markov chain instead of MDP")
      ("hybrid-verifier", po::value<string>()->default_value("hsolver"),
       "hybrid verifier to use (hsolver, phaver)")
      ("min-prob", po::value<bool>()->default_value(false),
       "minimise probabilities")
      ;
    
    po::options_description cmdline_options;
    cmdline_options.add(visible);
    
    /* parse options */
    try {
      po::store(po::command_line_parser(argc, argv).
		options(cmdline_options).run(), vm);
      po::notify(vm);
    } catch (...) {
      cerr << "Error on parsing command line." << endl;
      cerr << "Enter \"" << argv[0] << " --help\" for further help." << endl;
      exit(EXIT_FAILURE);
    }
    
    /* give help message if requested or wrong syntax used */
    if (vm.count("help")) {
      showHelp(argv[0], visible);
      exit(EXIT_SUCCESS);
    }
  }

}
  
