#ifndef GUARD_continuous_set_h
#define GUARD_continuous_set_h
/***************************************************************************
 *   Copyright (C) 2005 by Goran Frehse   *
 *   goran.frehse@imag.fr   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <vector>
#include <set>
#include <map>
#include "myPFunction.h"  // for remapping variables
#include "variable_id_map.h"  // for remapping variables

/*
   *  This class defines the base class for represenations of continuous sets.
   *  In principle, it keeps track of 
   *  The base class keeps track of string identifiers for the variables and
   *  and calls virtual functions with ending "_noids" that implement the
   *  semantic operations.
   *
   *  It is used to define continuous relations.
   *  
*/

enum continuous_set_type { polyhedron_exact, convex_polyhedron_exact, linear_constraint_exact }; 

class continuous_set
{
   public:
   // Constructors
   virtual continuous_set* clone() const;
   virtual continuous_set* construct_universe(dimension_t d, const variable_id_map_ptr pnew_map = variable_id_map_ptr() );
   virtual continuous_set* construct_empty(dimension_t d, const variable_id_map_ptr pnew_map = variable_id_map_ptr() );
   // Destructor
   virtual ~continuous_set() {}; 

   // Non-modifying non-semantic methods
   virtual int get_memory() const;
	virtual void print(ostream& s,int format_id) const = 0;

   // Non-modifying semantic methods
	virtual dimension_t get_dimension() const = 0;
	virtual dimension_t get_occupied_dimension() const;
   virtual bool is_empty() const = 0;
   virtual bool is_universe() const; // default implementation : complement is empty
   virtual bool is_disjoint_from(const continuous_set* cvs) const; // default implementation : intersection is empty
	virtual bool contains(const continuous_set* cvs) const; // default implementation : difference assign and emptiness
	// Linear programming
//	virtual bool minimize(const Linear_Expression &expr, Integer &inf_n, Integer &inf_d, bool &minimum) const = 0;
//   virtual bool maximize(const Linear_Expression &expr, Integer &sup_n, Integer &sup_d, bool &maximum) const; // default implementation: minimize (-expr,...)
  
   // Methods for modifying dimensions
   virtual void map_space_dimensions( const PFunction& pfunc, const variable_id_map_ptr pnew_map = variable_id_map_ptr() ); // pnew_map provides information in case new ids occur in the map
//   virtual void map_space_dimensions( const variable_id_map_ptr pnew_map ); // remap variables such that variables and dimension match pnew_map
   virtual void dimension_move_assign( const dimension_t& x1, const dimension_t& x2, const dimension_t& y, variable_id_map_ptr pnew_map = variable_id_map_ptr() ); // move vars x1--x2 to start t y
   virtual void add_space_dimensions_and_embed( dimension_t m, const variable_id_map_ptr pnew_map = variable_id_map_ptr() );
  	virtual void add_space_dimensions_and_embed_before( dimension_t m, const variable_id_map_ptr pnew_map = variable_id_map_ptr() );
   virtual void add_space_dimensions_and_embed_double( dimension_t m, const variable_id_map_ptr pnew_map = variable_id_map_ptr() );
   virtual void add_space_dimensions_and_project( dimension_t m, const variable_id_map_ptr pnew_map = variable_id_map_ptr() ); // new dimensions are initialized to zero

   virtual void remove_space_dimensions( const variable_ref_set& to_be_removed);
   virtual void remove_space_dimensions( const dimension_t& x1, const dimension_t& x2);
   virtual void remove_space_dimensions( const variable_name_set& to_be_removed);

   virtual void concatenate_assign( const continuous_set* cvs); // produce P x Q, where P=[this] and Q=cvs

   // Modifying methods with 0 continuous_set argument
   virtual void assign_variable_names( const variable_id_map_ptr pnew_map );
   virtual continuous_set* get_convex_hull() const = 0;
   virtual void simplify() = 0;
   virtual continuous_set* get_complement() const = 0; // cant' create default implementation (difference assign with universe) because type of universe is not known
   
   // Modifying methods with 1 continuous_set argument
	virtual void swap(continuous_set& cvs);
   virtual void time_elapse_assign(const continuous_set* tp) = 0;
   bool operator==(const continuous_set& cs) const; // call is_equal(this,pcs), which has default based on mutual containment
   virtual continuous_set* get_intersection(const continuous_set* p1) const; // caller function for get_intersection(this,p2)
   virtual void intersection_assign(const continuous_set* p1); // caller function for this = get_intersection(this,p2)
   virtual continuous_set* get_union(const continuous_set* p1) const; // caller function for get_intersection(this,p2)
   virtual void union_assign(const continuous_set* p1); // caller function for this = get_intersection(this,p2)
   virtual continuous_set* get_difference(const continuous_set* p1) const; // caller function for get_intersection(this,p2)
   virtual void difference_assign(const continuous_set* p1); // caller function for this = get_intersection(this,p2)
   
   protected:
   // Methods that don't take into account variable_id_map
   virtual continuous_set* clone_noids() const = 0;
   virtual continuous_set* construct_universe_noids(dimension_t d) = 0;
   virtual continuous_set* construct_empty_noids(dimension_t d) = 0;
   virtual void swap_noids(continuous_set& cvs) = 0;
   virtual void map_space_dimensions_noids( const PFunction& pfunc) = 0; 
   virtual void add_space_dimensions_and_embed_noids( dimension_t m );
   virtual void add_space_dimensions_and_project_noids( dimension_t m ) = 0; // new dimensions are initialized to zero
   virtual void remove_space_dimensions_noids( const variable_ref_set& to_be_removed) = 0;
   
   void set_pmy_variable_id_map( const variable_id_map_ptr pnew_map ); // no checks for consistency
   continuous_set_type my_type;
   variable_id_map_ptr pmy_variable_id_map;
};

// Operators that possibly take as arguments two continuous_sets of different derived classes

   continuous_set* get_union_impl(const continuous_set* p1, const continuous_set* p2);
   continuous_set* get_intersection_impl(const continuous_set* p1, const continuous_set* p2);
   continuous_set* get_difference_impl(const continuous_set* p1, const continuous_set* p2);

#endif
