/***************************************************************************
 *   Copyright (C) 2004 by Goran Frehse                                    *
 *   gfrehse@localhost                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
 
#include "myPFunction.h"

using namespace Parma_Polyhedra_Library;
using namespace IO_Operators;
using namespace std;
 
// --------------------------------------------------------------

// Mapping function for shifting dimensions of ccvs and cvs
// taken from PFunction.hh

ostream&
operator<<( ostream& os, const PFunction &pf )
    {
      pf.print(os);
      return os;
    }

  PFunction::PFunction() : mymax(0)
    {}

  void
  PFunction::swap_assign(dim_t x1, dim_t x2, dim_t y, dim_t n)
  {
    // swaps the variables in positions x1 ... x2 with y ... y2=x2-x1+y)
    // in a space of size n
    mymap.clear();
    // initialize map
    for (dim_t i=0;i<n;++i)
    {
      mymap.insert(Map::value_type(i, i));
    };
    for (dim_t x=x1;x<=x2;++x)
    {
      mymap[x]=y;
      mymap[y]=x;
      ++y;
    };
    mymax=n-1;
  }

  void
  PFunction::move_assign(dim_t x1, dim_t x2, dim_t y, dim_t n)
  {
    // moves the variables x1 ... x2 to position y in a map of size n of the original state set
    // swaps the variables in positions x1 ... x2 with y ... y2=x2-x1+y)
    mymap.clear();
    // initialize map
    for (dim_t i=0;i<n;++i)
    {
      mymap.insert(Map::value_type(i, i));
    };
    // move the x values to the position of y
    dim_t y1=y;
    for (dim_t x=x1;x<=x2;++x)
    {
      mymap[x]=y1;
      ++y1;
    };
    if (y>=x1)
    {
      // shift the remaining ones to the left
      y1=x1;
      for (dim_t i=x2+1;i+x1<=y+x2;++i)
      {
        mymap[i]=y1;
        ++y1;
      };
    }
    else
    {
      // shift the remaining ones to the right
      y1=y+x2-x1+1;
      for (dim_t i=y;i<x1;++i)
      {
        mymap[i]=y1;
        ++y1;
      };
    };
    mymax=n-1;
  }

  bool
  PFunction::has_empty_codomain() const {
    return mymap.empty();
  }

  Parma_Polyhedra_Library::dimension_type PFunction::max_in_codomain() const {
    if (has_empty_codomain())
      throw std::runtime_error("PFunction::max_in_codomain() called when has_empty_codomain()");
    return mymax;
  }

  bool
  PFunction::maps(dim_t x, dim_t& y) const {
    if (has_empty_codomain())
      throw std::runtime_error("PFunction::maps() called"
  			     " when has_empty_codomain()");
    Map::const_iterator i = mymap.find(x);
    if (i != mymap.end()) {
      y = (*i).second;
      return true;
    }
    else
      return false;
  }

//  dim_t
	Parma_Polyhedra_Library::dimension_type // NOTE:: dim_t yiels compilation error!
  PFunction::get_map(dim_t x) const {
    if (has_empty_codomain())
      throw std::runtime_error("PFunction::get_map() called"
  			     " when has_empty_codomain()");
    Map::const_iterator i = mymap.find(x);
    if (i==mymap.end())
      throw std::runtime_error("PFunction::get_map() called"
  			     " with non-existent x");
    return i->second;
  }
  
Parma_Polyhedra_Library::dimension_type // NOTE:: dim_t yiels compilation error!
  PFunction::get_premap(dim_t y) const {
    if (mymap.empty())
      throw std::runtime_error("PFunction::get_premap() called"
  			     " when map is empty");
    for (Map::const_iterator i = mymap.begin();i!=mymap.end();++i)
    {
      if (i->second==y)
        return i->first;
    };
      throw std::runtime_error("PFunction::get_premap() called "
  			     " with non-existent y");    
    return mymap.begin()->first; // just return a dummy value
  }  

  bool
  PFunction::in_domain(const dim_t& x) const {
    for (Map::const_iterator i = mymap.begin();i!=mymap.end();++i)
    {
      if (i->first==x)
        return true;
    };
    return false;
  }
    
  bool
  PFunction::in_codomain(const dim_t& y) const {
    for (Map::const_iterator i = mymap.begin();i!=mymap.end();++i)
    {
      if (i->second==y)
        return true;
    };
    return false;
  }
  
  void
  PFunction::fill_up_to(dim_t newdim)
  {
    // add the remaining variables that are not already mapped by *this to map
    // in no particular order (first free slots)
    // (they have to be in it, because otherwise they will be removed)
    // just fill in the slots that are not yet taken

    for (dimension_type i=0;i<newdim;++i)
    {
//      if (!in_domain(i)) // map those that are not yet in the map
      if (!in_codomain(i)) // map those that are not yet in the map
      {
        for (dimension_type jj=0;jj<newdim;++jj)
        {
//          if (!in_codomain(jj))
          if (!in_domain(jj))
          {
//            insert(i,jj);
            insert(jj,i);
            break;
          };
        };
      };
    };
  }

  void
  PFunction::print(std::ostream& s) const {
    using namespace Parma_Polyhedra_Library;
    using namespace IO_Operators;

    if (has_empty_codomain())
      s << "empty" << std::endl;
    else
      for (Map::const_iterator i = mymap.begin(); i != mymap.end(); ++i)
        s << Variable((*i).first) << " --> "
  	<< Variable((*i).second)
  	<< std::endl;
  }

  void
  PFunction::insert(dim_t x, dim_t y) {
//	mymap[x]=y;
   std::pair<Map::iterator, bool> stat
     = mymap.insert(Map::value_type(x, y));
   if (!stat.second)
     throw_error("PFunction::insert(x, y) called with `x' already in domain");
   if (y > mymax)
     mymax = y;
  }
