



#include "DHR_split.h"
#include "general.h"

DHR_Rational** getRangeDot(C_Polyhedron* polyhedron, DHR_Rational** coeff, DHR_Rational* bound)
{
	DHR_Rational** rangeDot = new DHR_Rational*[2];
	rangeDot[0] = new DHR_Rational[2];
	rangeDot[1] = new DHR_Rational[2];
	for(int i=0; i<2; i++)
	{
		Linear_Expression expr = Linear_Expression(bound[i][0] * coeff[i][0][1] * coeff[i][1][1] ); //bound[i];
		expr += (bound[i][1] * coeff[i][0][0] * coeff[i][1][1]) * Variable(0);
		expr += (bound[i][1] * coeff[i][0][1] * coeff[i][1][0]) * Variable(1);

		rangeDot[i][0] = new Integer[2];
		rangeDot[i][1] = new Integer[2];
		bool isMin;
		polyhedron->minimize(expr, rangeDot[i][0][0], rangeDot[i][0][1], isMin);
		bool isMax;
		polyhedron->maximize(expr, rangeDot[i][1][0], rangeDot[i][1][1], isMax);
		assert(isMin && isMax && (rangeDot[i][0][1] > 0) && (rangeDot[i][1][1] > 0));
		rangeDot[i][0][1] *= (bound[i][1] * coeff[i][0][1] * coeff[i][1][1]);
		rangeDot[i][1][1] *= (bound[i][1] * coeff[i][0][1] * coeff[i][1][1]);
		simplifyRat(rangeDot[i][0]);
		simplifyRat(rangeDot[i][1]);
	}
	//displayRange(rangeDot, 'c');
	return rangeDot;
}

DHR_Rational** getRange(C_Polyhedron* polyhedron)
{
	DHR_Rational** range = new DHR_Rational*[2];
	range[0] = new DHR_Rational[2];
	range[1] = new DHR_Rational[2];
	for(uint i=0; i<polyhedron->space_dimension(); i++)
	{
		Linear_Expression expr = Linear_Expression(0); //bound[i];
		expr += Variable(0);

		range[i][0] = new Integer[2];
		range[i][1] = new Integer[2];
		bool isMin;
		polyhedron->minimize(expr, range[i][0][0], range[i][0][1], isMin);
		bool isMax;
		polyhedron->maximize(expr, range[i][1][0], range[i][1][1], isMax);
		assert(isMin && isMax && (range[i][0][1] > 0) && (range[i][1][1] > 0)); // denominators strictly positive
		simplifyRat(range[i][0]);
		simplifyRat(range[i][1]);
	}
	//displayRange(rangeDot, 'c');
	return range;
}

DHR_Rational getVolumeRange(DHR_Rational** range)
{
  // range must be an array of size 2
	DHR_Rational r1 = subRat(range[0][1],range[0][0]);
	DHR_Rational r2 = subRat(range[1][1],range[1][0]);
	DHR_Rational result = mulRat(r1,r2);
	delete[] r1;
	delete[] r2;
	simplifyRat(result);
	return result;
}

DHR_Rational getSizeRange(DHR_Rational** range)
{
	DHR_Rational r1 = subRat(range[0][1],range[0][0]);
	DHR_Rational r2 = subRat(range[1][1],range[1][0]);
	DHR_Rational result = maxRat(r1,r2);
	delete[] r1;
	delete[] r2;
	simplifyRat(result);
	return result;
}

void deleteRange(DHR_Rational** range)
{
	for(int i=0; i<2; i++)
	{
		delete[] range[i];
	}
	delete[] range;
}

DHR_Rational** getQMinMax(C_Polyhedron* polyhedron, int minR, int maxR, DHR_Rational* rangeMinVariable, DHR_Rational** coeff, DHR_Rational* bound)
{
	DHR_Rational** Q = new DHR_Rational*[2];
	Q[0] = new DHR_Rational[2];
	Q[1] = new DHR_Rational[2];

	Linear_Expression goal = Linear_Expression(bound[maxR][0] * coeff[maxR][0][1] * coeff[maxR][1][1]);
	goal += (bound[maxR][1] * coeff[maxR][0][0] * coeff[maxR][1][1]) * Variable(0);
	goal += (bound[maxR][1] * coeff[maxR][0][1] * coeff[maxR][1][0]) * Variable(1);
	Integer goal_denominator = bound[maxR][1] * coeff[maxR][0][1] * coeff[maxR][1][1];
	
	//cout << *polyhedron << endl;
	//displayRat(rangeMinVariable[0]); cout << " , ";
	//displayRat(rangeMinVariable[1]); cout << endl;

	for(int i=0; i<2; i++)
	{
		C_Polyhedron* polyhedron2D = new C_Polyhedron(*polyhedron);
		DHR_Rational  freeTerm = subRat(bound[minR], rangeMinVariable[i]);
		simplifyRat(freeTerm);
		Linear_Expression expr = Linear_Expression(freeTerm[0] * coeff[minR][0][1] * coeff[minR][1][1]);
		expr += (freeTerm[1] * coeff[minR][0][0] * coeff[minR][1][1]) * Variable(0);
		expr += (freeTerm[1] * coeff[minR][0][1] * coeff[minR][1][0]) * Variable(1);
		polyhedron2D->add_constraint(expr == 0);
		Q[i][0] = new Integer[2];
		Q[i][1] = new Integer[2];
		
		bool isMin;
		polyhedron2D->minimize(goal, Q[i][0][0], Q[i][0][1], isMin);
		bool isMax;
		polyhedron2D->maximize(goal, Q[i][1][0], Q[i][1][1], isMax);

		/*
		cout << "-------------------- " << i << " ---------------------" << endl;
		cout << *(polyhedron2D) << endl;
		cout << goal << endl;
		cout << isMin << endl;
		cout << isMax << endl;
		cout << Q[i][0][0] << "/" << Q[i][0][1] << endl;
		cout << Q[i][1][0] << "/" << Q[i][1][1] << endl;
		cout << "--------------------------------------------" << endl;
		*/
		assert(isMin && isMax && (Q[i][0][1] > 0) && (Q[i][1][1] > 0));
		Q[i][0][1] *= goal_denominator;
		Q[i][1][1] *= goal_denominator;
		simplifyRat(Q[i][0]);
		simplifyRat(Q[i][1]);
		delete[] freeTerm;
		delete polyhedron2D;
	}
	return Q;
}

DHR_Rational* getAllDelta(C_Polyhedron* polyhedron, DHR_Rational** coeff, DHR_Rational* bound, DHR_Rational** rangeDot, DHR_Rational delta_0, int maxR)
{
	DHR_Rational* delta_Z = new DHR_Rational[4];

	DHR_Rational* startPosition = new DHR_Rational[2];
	int* direction = new int[2];

	if (maxR == 0) {startPosition[0] = addRat(rangeDot[0][0], delta_0); startPosition[1] = copyRat(rangeDot[1][0]);}
	else {startPosition[0] = copyRat(rangeDot[0][0]); startPosition[1] = addRat(rangeDot[1][0], delta_0);}
	direction[0] = 1;
	direction[1] = 1;
	delta_Z[0] = getMinDelta(polyhedron, startPosition, direction, coeff, bound);   // delta_A

	if (maxR == 0) {startPosition[0] = addRat(rangeDot[0][0], delta_0); startPosition[1] = copyRat(rangeDot[1][1]);}
	else {startPosition[0] = copyRat(rangeDot[0][0]); startPosition[1] = subRat(rangeDot[1][1], delta_0);}
	direction[0] = 1;
	direction[1] = -1;
	delta_Z[1] = getMinDelta(polyhedron, startPosition, direction, coeff, bound);   // delta_B

	if (maxR == 0) {startPosition[0] = subRat(rangeDot[0][1], delta_0); startPosition[1] = copyRat(rangeDot[1][1]);}
	else {startPosition[0] = copyRat(rangeDot[0][1]); startPosition[1] = subRat(rangeDot[1][1], delta_0);}
	direction[0] = -1;
	direction[1] = -1;
	delta_Z[2] = getMinDelta(polyhedron, startPosition, direction, coeff, bound);   // delta_C

	if (maxR == 0) {startPosition[0] = subRat(rangeDot[0][1], delta_0); startPosition[1] = copyRat(rangeDot[1][0]);}
	else {startPosition[0] = copyRat(rangeDot[0][1]); startPosition[1] = addRat(rangeDot[1][0], delta_0);}
	direction[0] = -1;
	direction[1] = 1;
	delta_Z[3] = getMinDelta(polyhedron, startPosition, direction, coeff, bound);   // delta_D

	return delta_Z;
}

DHR_Rational getMinDelta(C_Polyhedron* polyhedron, DHR_Rational* startPosition, int* direction, DHR_Rational** coeff, DHR_Rational* bound)
{
	C_Polyhedron* polyhedron2D = new C_Polyhedron(*polyhedron);
	assert(polyhedron2D->space_dimension() == 2);
	polyhedron2D->add_space_dimensions_and_embed(1);
	DHR_Rational delta = new Integer[2];	
	for(int i=0; i<2; i++) 
	{
		DHR_Rational freeTerm = subRat(bound[i], startPosition[i]);
		simplifyRat(freeTerm);
		Linear_Expression expr = Linear_Expression(freeTerm[0] * coeff[i][0][1] * coeff[i][1][1] );
		expr += (freeTerm[1] * coeff[i][0][0] * coeff[i][1][1]) * Variable(0);
		expr += (freeTerm[1] * coeff[i][0][1] * coeff[i][1][0]) * Variable(1);
		expr -= (direction[i] * freeTerm[1] * coeff[i][0][1] * coeff[i][1][1]) * Variable(2);
		polyhedron2D->add_constraint(expr == 0);
		Linear_Expression goal = Variable(2);
		bool isMin;
		polyhedron2D->minimize(goal, delta[0], delta[1], isMin);
		delete[] freeTerm;
	}
	return delta;
	
}


// return the intersection of lines L1: a1x+b1y=c1 and L2: a2x+b2y=c2
DHR_Rational* intersect(DHR_Rational a1, DHR_Rational b1, DHR_Rational c1, DHR_Rational a2, DHR_Rational b2, DHR_Rational c2)
{
	DHR_Rational* result = new DHR_Rational[2];
	bool emptyIntersection = false;

	if (DHR_output_level >= 2) 
	{
		cout << "Intersection of Line: ";
		displayRat(a1);	cout << "x + ";
		displayRat(b1);	cout << "y = ";
		displayRat(c1);	cout << " and ";
		displayRat(a2);	cout<< "x + ";
		displayRat(b2);	cout << "y = ";
		displayRat(c2);	cout << endl;
	}

	DHR_Rational a1b2 = mulRat(a1,b2);
	DHR_Rational a2b1 = mulRat(a2,b1);
	DHR_Rational det = subRat(a1b2, a2b1);
	if (det[0] != 0)
	{
		DHR_Rational a1c2 = mulRat(a1,c2);
		DHR_Rational a2c1 = mulRat(a2,c1);
		DHR_Rational b2c1 = mulRat(b2,c1);
		DHR_Rational b1c2 = mulRat(b1,c2);
		DHR_Rational difAC = subRat(a1c2, a2c1);
		DHR_Rational difBC = subRat(b2c1, b1c2);
		result[1] = divRat(difAC, det);
		result[0] = divRat(difBC, det);
		simplifyRat(result[0]);
		simplifyRat(result[1]);
		delete[] a1c2; delete[] a2c1;
		delete[] b2c1; delete[] b1c2;
		delete[] difAC; delete[] difBC;
	}
	else emptyIntersection = true;

	if (emptyIntersection) {cerr << "Error !! Empty intersection." << endl; exit(1);}

	if (DHR_output_level >= 2) 
	{
		cout << " (";
		displayRat(result[0]);
		cout << ",";
		displayRat(result[1]);
		cout << ")" << endl;
	}
	return result;
}

// returns the line: ax+by=c passing by P1 and P2
// P1 = intersect (f1 = level[0][0]) and (f2 = level[1][0])
// P2 = intersect (f1 = level[0][1]) and (f2 = level[1][1])
// f1 = coeff[0][0] x + coeff[0][1] y + bound[0]
// f2 = coeff[1][0] x + coeff[1][1] y + bound[1]
void getLine(DHR_Rational** coeff, DHR_Rational* bound, DHR_Rational** level, DHR_Rational& a, DHR_Rational& b, DHR_Rational& c)
{
	DHR_Rational* P1;
	DHR_Rational freeTerm1 = subRat(level[0][0], bound[0]);
	DHR_Rational freeTerm2 = subRat(level[1][0], bound[1]);
	simplifyRat(freeTerm1);
	simplifyRat(freeTerm2);
	P1 = intersect(coeff[0][0], coeff[0][1], freeTerm1, coeff[1][0], coeff[1][1], freeTerm2);
	delete[] freeTerm1;
	delete[] freeTerm2;
	DHR_Rational* P2;
	freeTerm1 = subRat(level[0][1], bound[0]);
	freeTerm2 = subRat(level[1][1], bound[1]);
	simplifyRat(freeTerm1);
	simplifyRat(freeTerm2);
	P2 = intersect(coeff[0][0], coeff[0][1], freeTerm1, coeff[1][0], coeff[1][1], freeTerm2);
	delete[] freeTerm1;
	delete[] freeTerm2;

	a=subRat(P2[1], P1[1]);
	b=subRat(P1[0], P2[0]);
	DHR_Rational r1 = mulRat(a, P1[0]);
	DHR_Rational r2 = mulRat(b, P1[1]);
	c=addRat(r1,r2);
	simplifyRat(a); simplifyRat(b); simplifyRat(c);
	delete[] r1; delete[] r2;
	if (DHR_output_level >= 2) 
	{
		cout << "Line passing by ("; displayRat(P1[0]);	cout << ","; displayRat(P1[1]);
		cout << ") and ("; displayRat(P2[0]); cout << ","; displayRat(P2[1]); cout << "): ";
		displayRat(a); cout << "x + "; displayRat(b); cout << "y = "; displayRat(c); cout << endl;
	}
	delete[] P1[0]; delete[] P1[1];
	delete[] P2[0]; delete[] P2[1];
	delete[] P1; delete[] P2;
}


// Line returned: ax+by=c;
// result[0] = a
// result[1] = b
// result[2] = c
DHR_Rational* getOptimalCut(C_Polyhedron* polyhedron, DHR_Rational** rangeDot, DHR_Rational** coeff, DHR_Rational* bound)
{
	const int HALF_CUT = 1;
	const int LINE_A_C = 2;
	const int LINE_B_D = 3;


	int typeResult = 0;
	DHR_Rational* result = new DHR_Rational[3];
	//DHR_Rational** rangeDot = getRangeDot(polyhedron, coeff, bound);
	DHR_Rational* rangeSize = new DHR_Rational[2];
	rangeSize[0] = subRat(rangeDot[0][1],rangeDot[0][0]);
	rangeSize[1] = subRat(rangeDot[1][1],rangeDot[1][0]);
	simplifyRat(rangeSize[0]); simplifyRat(rangeSize[1]);

  //DHR_Rational** range = getRange(polyhedron);

	// Check if more splitting is needed
	if (gtRat(DHR_THRESHOLD, rangeSize[0]) && (gtRat(DHR_THRESHOLD, rangeSize[1])))
  //if (gtRat(DHR_THRESHOLD, mulRat(getVolumeRange(range),getVolumeRange(rangeDot))))
	{
		//cerr << "No refinement necessary !! (THRESHOLD = "; displayRat(DHR_THRESHOLD); cerr  << endl;
		//int q; cin >> q;
		for(int i=0; i<3; i++) {result[i] = new Integer[2]; result[i][0] = 0; result[i][1] = 1;}
//    deleteRange(range);
		return result;
	}
//  deleteRange(range);
	int maxR,minR;
 	if (gtRat(rangeSize[0], rangeSize[1])) {maxR=0, minR=1;}
	else {maxR=1, minR=0;}

	// DEBUG
	if (DHR_output_level >= 2) 
	{
		cout << endl; displayRange(rangeDot, 'z'); 
		cout << " Max range size: "; displayRat(rangeSize[maxR]); cout << endl;
		cout << " Min range size: "; displayRat(rangeSize[minR]); cout << endl << endl;
	}

	DHR_Rational twiceMinR = copyRat(rangeSize[minR]);
	twiceMinR[0] *= 2;  // multiplication by 2
	DHR_Rational halfMaxR = copyRat(rangeSize[maxR]);
	halfMaxR[1] *= 2;  // division by 2

	if (geqRat(rangeSize[maxR], twiceMinR))
	{
		typeResult = HALF_CUT;
		if (DHR_output_level >= 2) cout << "Line 6 end. (Half-Cut)" << endl;
	}
	else
	{
		DHR_Rational delta_0 = subRat(rangeSize[maxR], rangeSize[minR]);
		simplifyRat(delta_0);
		if (DHR_output_level >= 2) {cout << "delta_0 = "; displayRat(delta_0); cout << endl;}

		DHR_Rational* delta_Z = getAllDelta(polyhedron, coeff, bound, rangeDot, delta_0, maxR);
		DHR_Rational delta_A = delta_Z[0]; // Warning: this is NOT a copy...
		DHR_Rational delta_B = delta_Z[1];
		DHR_Rational delta_C = delta_Z[2];
		DHR_Rational delta_D = delta_Z[3];

		// DEBUG
		if (DHR_output_level >= 2) 
		{
			cout << "delta_A = "; displayRat(delta_A); cout << endl;
			cout << "delta_B = "; displayRat(delta_B); cout << endl;
			cout << "delta_C = "; displayRat(delta_C); cout << endl;
			cout << "delta_D = "; displayRat(delta_D); cout << endl;
		}

		DHR_Rational delta1 = minRat(delta_A, delta_C);
		DHR_Rational delta2 = minRat(delta_B, delta_D);

		DHR_Rational rTemp = subRat(halfMaxR, delta_0);
		if (geqRat(delta1, rTemp) && geqRat(delta2, rTemp))
		{
			typeResult = HALF_CUT;
			if (DHR_output_level >= 2) cout << "Line 17 end. (Half-Cut)" << endl;
		}
		else
		{
			assert(gtRat(rTemp, delta1) && gtRat(rTemp, delta2));
			if (DHR_output_level >= 2) cout << " Calling getQMinMax...\n" << "minR = " << minR << "; maxR = " << maxR << endl;
			DHR_Rational ** rangeQ = getQMinMax(polyhedron, minR, maxR, rangeDot[minR], coeff, bound);
			if (DHR_output_level >= 2) displayRange(rangeQ, 'Q');
			DHR_Rational lower = addRat(rangeDot[maxR][0], delta_0);
			DHR_Rational upper = subRat(rangeDot[maxR][1], delta_0);
			if (geqRat(lower, rangeQ[0][0]) && geqRat(rangeQ[0][1], upper))
			{
				typeResult = HALF_CUT;
				if (DHR_output_level >= 2) cout << "Line 21 end. (Half-Cut)" << endl;
			}
			else if (geqRat(lower, rangeQ[0][0]))
			{
				if (geqRat(lower, rangeQ[1][0]))
				{
					typeResult = HALF_CUT;
					if (DHR_output_level >= 2) cout << "Line 24 end. (Half-Cut)" << endl;
				}
				else // return line(B_delta2,D_delta2);
				{
					typeResult = LINE_B_D;
					if (DHR_output_level >= 2) cout << "Line 25 end." << endl;
				}
			}
			else if (geqRat(rangeQ[0][1], upper))
			{
				if (geqRat(rangeQ[1][1], upper))
				{
					typeResult = HALF_CUT;
					if (DHR_output_level >= 2) cout << "Line 28 end. (Half-Cut)" << endl;
				}
				else // return line(A_delta1,C_delta1);
				{
					typeResult = LINE_A_C;
					if (DHR_output_level >= 2) cout << "Line 29 end." << endl;
				}
			}
			else
			{
				if (geqRat(lower, rangeQ[1][0]) && geqRat(rangeQ[1][1], upper))
				{
					typeResult = HALF_CUT;
					if (DHR_output_level >= 2) cout << "Line 31 end. (Half-Cut)" << endl;
				}
				else if (geqRat(lower, rangeQ[1][0])) // return line(A_delta1,C_delta1);
				{
					typeResult = LINE_A_C;
					if (DHR_output_level >= 2) cout << "Line 33 end." << endl;
				}
				else if (geqRat(rangeQ[1][1], upper)) // return line(B_delta2,D_delta2);
				{
					typeResult = LINE_B_D;
					if (DHR_output_level >= 2) cout << "Line 35 end." << endl;
				}
				else if ((delta1[0] == 0) && (delta2[0] == 0))
				{
					typeResult = HALF_CUT;
					cerr << "Special case Line 36 end. (Half-Cut)" << endl;
					int q; cin >> q;
				}
				else if (gtRat(delta1,delta2))
				{
					typeResult = LINE_A_C;
					if (DHR_output_level >= 2) cout << "Line 37 end." << endl;
				}
				else
				{
					typeResult = LINE_B_D;
					if (DHR_output_level >= 2) cout << "Line 38 end." << endl;
				}
			}

			if (typeResult == LINE_A_C)
			{
				DHR_Rational** level = new DHR_Rational*[2];
				level[0] = new DHR_Rational[2];
				level[1] = new DHR_Rational[2];
				if(minR == 0)
				{
					level[0][0] = addRat(rangeDot[minR][0], delta1);
					level[0][1] = subRat(rangeDot[minR][1], delta1);
					level[1][0] = addRat(lower, delta1);
					level[1][1] = subRat(upper, delta1);
				}
				else
				{
					level[0][0] = addRat(lower, delta1);
					level[0][1] = subRat(upper, delta1);
					level[1][0] = addRat(rangeDot[minR][0], delta1);
					level[1][1] = subRat(rangeDot[minR][1], delta1);
				}
				getLine(coeff, bound, level, result[0], result[1], result[2]);
				deleteRange(level);
			}
			else if (typeResult == LINE_B_D)
			{
				DHR_Rational** level = new DHR_Rational*[2];
				level[0] = new DHR_Rational[2];
				level[1] = new DHR_Rational[2];
				if(minR == 0)
				{
					level[0][0] = addRat(rangeDot[minR][0],delta2);
					level[0][1] = subRat(rangeDot[minR][1],delta2);
					level[1][0] = subRat(upper,delta2);
					level[1][1] = addRat(lower,delta2);
				}
				else
				{
					level[0][0] = subRat(upper,delta2);
					level[0][1] = addRat(lower,delta2);
					level[1][0] = addRat(rangeDot[minR][0],delta2);
					level[1][1] = subRat(rangeDot[minR][1],delta2);
				}
				getLine(coeff, bound, level, result[0], result[1], result[2]);
				deleteRange(level);
			}

			deleteRange(rangeQ);
			delete[] lower;
			delete[] upper;
		}
		
		delete[] delta_Z[0]; delete[] delta_Z[1];
		delete[] delta_Z[2]; delete[] delta_Z[3];
		delete[] delta_Z;
		delete[] delta1;
		delete[] delta2;
		delete[] rTemp;
	}

	if (typeResult == HALF_CUT)
	{
		result[0] = copyRat(coeff[maxR][0]);
		result[1] = copyRat(coeff[maxR][1]);
		DHR_Rational rTemp = addRat(rangeDot[maxR][0], halfMaxR);
		result[2] = subRat(rTemp, bound[maxR]);
		delete[] rTemp;
	}

	delete[] rangeSize[0];
	delete[] rangeSize[1];
	delete[] rangeSize;
	delete[] twiceMinR;
	delete[] halfMaxR;
	return result;
}

// Line returned: ax+by=c;
// result[0] = a
// result[1] = b
// result[2] = c
DHR_Rational* getHalfCut(C_Polyhedron* polyhedron, DHR_Rational** rangeDot, DHR_Rational** coeff, DHR_Rational* bound)
{
	DHR_Rational* result = new DHR_Rational[3];

	DHR_Rational* rangeSize = new DHR_Rational[2];
	rangeSize[0] = subRat(rangeDot[0][1],rangeDot[0][0]);
	rangeSize[1] = subRat(rangeDot[1][1],rangeDot[1][0]);
	simplifyRat(rangeSize[0]); simplifyRat(rangeSize[1]);
	if (gtRat(DHR_THRESHOLD, rangeSize[0]) && (gtRat(DHR_THRESHOLD, rangeSize[1])))
	{
		cerr << "No refinement necessary !! (DHR_THRESHOLD = "; displayRat(DHR_THRESHOLD); cerr  << endl;
		int q; cin >> q;
		for(int i=0; i<3; i++) {result[i] = new Integer[2]; result[i][0] = 0; result[i][1] = 1;}
		return result;
	}
	int maxR;
 	if (gtRat(rangeSize[0], rangeSize[1])) 
	{
		maxR=0;	
	}
	else
	{
		maxR=1;	
	}
	DHR_Rational halfMaxR = copyRat(rangeSize[maxR]);
	halfMaxR[1] *= 2;  // division by 2

	result[0] = copyRat(coeff[maxR][0]);
	result[1] = copyRat(coeff[maxR][1]);
	DHR_Rational rTemp = addRat(rangeDot[maxR][0], halfMaxR);
	result[2] = subRat(rTemp, bound[maxR]);
	delete[] rTemp;
	delete[] halfMaxR;
	delete[] rangeSize[0];
	delete[] rangeSize[1];
	delete[] rangeSize;
	if (DHR_output_level >= 2) 
	{
		cout << "Half-Cut line: "; displayRat(result[0]); cout << "x + ";
		displayRat(result[1]); cout << "y = "; displayRat(result[2]); cout << endl;
	}
	return result;
}


void splitPolygon(DHR_PolyhedronSet* polyhedronSet, DHR_Rational** coeff, DHR_Rational* bound)
{
	if (DHR_output_level >= 2) 
	{
		cout << "Polygon to cut: " << endl;
		cout << *(polyhedronSet->polyhedron) << endl;
	}

	//C_Polyhedron* polyhedron2D = new C_Polyhedron(*(polyhedronSet->polyhedron));
	DHR_Rational* lineCut;
	if (DHR_HALF_CUT_METHOD)
	{
		lineCut = getHalfCut(polyhedronSet->polyhedron, polyhedronSet->sizeRange, coeff, bound);
	}
	else
	{
		lineCut = getOptimalCut(polyhedronSet->polyhedron, polyhedronSet->sizeRange, coeff, bound);
	}

	if ((lineCut[0][0] != 0) || (lineCut[1][0] != 0) || (lineCut[2][0] != 0))
	{
		C_Polyhedron* newPolyhedron = new C_Polyhedron(*(polyhedronSet->polyhedron));
		polyhedronSet->polyhedron->add_constraint(
		(lineCut[0][0] * lineCut[1][1] * lineCut[2][1]) * Variable(0) + 
		(lineCut[0][1] * lineCut[1][0] * lineCut[2][1]) * Variable(1) <=
		lineCut[0][1] * lineCut[1][1] * lineCut[2][0]);
		assert(!(polyhedronSet->polyhedron->is_empty()));
		deleteRange(polyhedronSet->sizeRange);
		delete[] polyhedronSet->maxSizeRange;
		polyhedronSet->sizeRange = getRangeDot(polyhedronSet->polyhedron, coeff, bound);
		polyhedronSet->maxSizeRange = getSizeRange(polyhedronSet->sizeRange);
		newPolyhedron->add_constraint(
		(lineCut[0][0] * lineCut[1][1] * lineCut[2][1]) * Variable(0) + 
		(lineCut[0][1] * lineCut[1][0] * lineCut[2][1]) * Variable(1) >=
		lineCut[0][1] * lineCut[1][1] * lineCut[2][0]);
		assert(!(newPolyhedron->is_empty()));
		polyhedronSet->add(newPolyhedron, coeff, bound);
	}
	delete[] lineCut[0];
	delete[] lineCut[1];
	delete[] lineCut[2];
	delete[] lineCut;
	//polyhedronSet->display();
	//int q; cin >> q;
}

/*
	IloNumVar x(env, 0.0, IloInfinity, ILOFLOAT, "x");
	IloNumVar y(env, 0.0, IloInfinity, ILOFLOAT, "y");
	IloModel model(env);
	model.add(y);
	try
	{
		model.add(x+y <= 1);
		model.add(x+y >= -1);
		model.add(x-y <= 1);
		model.add(x-y >= -1);
	} 
	catch (IloException& ex)
	{
		cerr << "Error: " << ex << endl;
	}
	IloCplex cplex(model);
	cplex.exportModel("polygon.lp");
	system("cat polygon.lp");

	model.remove(y);

	IloCplex cplex2(model);
	cplex2.exportModel("polygon2.lp");
	system("cat polygon2.lp");
*/

/*


void splitPolygon(IloEnv env, PolygonSet polygonSet, IloNumArray2 coeff, IloNumArray bound)
{
	IloInt nVar=polygonSet[0][0].getSize() - 1;
	IloInt k;

	IloNum bestCut=-1;
	IloInt polygonId, varId, cutId, totCut;

	IloEnv workingEnv;

	for(k=0; k<polygonSet.getSize(); k++)    // consider each polygon of the set
	{
		workingEnv.out() << "**************************************************" << endl;
		workingEnv.out() << "Considering polygon #" << k << "..." << endl;
		workingEnv.out() << "**************************************************" << endl;
		assert(nVar == polygonSet[k][0].getSize() - 1);


		IloNumArray2 range = getRangeDot(workingEnv, polygonSet[k], coeff, bound);

		displayRange(workingEnv, range);

		IloIntArray nCut(workingEnv);
		nCut.add(3); nCut.add(3);
		IloInt i,j;

		for(i=0; i<nVar; i++)   // consider cutting along each variable
		{
			workingEnv.out() << "**************************************************" << endl;
			workingEnv.out() << "   Cutting along variable  #" << i << "..." << endl;
			workingEnv.out() << "**************************************************" << endl;
			IloNumArray cutExpr(workingEnv);
			IloInt q;
			for(q=0; q<nVar; q++)
			{
				cutExpr.add(coeff[i][q]);
			}
			cutExpr.add(0);
			IloNumArray cutExpr2(workingEnv);
			for(q=0; q<nVar; q++)
			{
				cutExpr2.add(-coeff[i][q]);
			}
			cutExpr2.add(0);

			IloNum min = range[i][0];
			IloNum max = range[i][1];

			//workingEnv.out() << " ### Checking range for variable #" << i << endl;
			//workingEnv.out() << "     Min = " << min << "     Max = " << max << endl;
			//workingEnv.out() << " ### ### ### ### ### ### ### ### ###" << endl;
			assert (nCut[i] >= 2);
			for(j=1; j<nCut[i]; j++)   // consider each cut on the grid, keep the best
			{
				//workingEnv.out() << "**************************************************" << endl;
				//workingEnv.out() << "      Cutting nbr #" << j << "/" << nCut[i] << "..." << endl;
				//workingEnv.out() << "**************************************************" << endl;
				IloNum cut = min + (max - min) * j / nCut[i];
				//workingEnv.out() << "     Cut = " << cut << endl;
				cutExpr[cutExpr.getSize()-1] = cut - bound[i];
				polygonSet[k].add(cutExpr);
	
				//displayPolygon(workingEnv, polygonSet[k]);

				IloNumArray2 range1 = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
				IloNum area1 = getArea(workingEnv, polygonSet[k]);
				IloNum thisCut = area1*(range1[0][0] - range1[0][1] + range1[1][0] - range1[1][1]);

				polygonSet[k].remove(polygonSet[k].getSize()-1);
				cutExpr2[cutExpr2.getSize()-1] = bound[i] - cut;
				polygonSet[k].add(cutExpr2);

				//displayPolygon(workingEnv, polygonSet[k]);
				
				IloNumArray2 range2 = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
				
				IloNum area2 = getArea(workingEnv, polygonSet[k]);
				thisCut += area2*(range2[0][0] - range2[0][1] + range2[1][0] - range2[1][1]);
				polygonSet[k].remove(polygonSet[k].getSize()-1);


				displayRange(workingEnv, range);
				displayRange(workingEnv, range1);
				displayRange(workingEnv, range2);
				
				IloNum before = (area1 + area2) * (range[0][1] - range[0][0] + range[1][1] - range[1][0]);
				IloNum after = area1 * (range1[0][1] - range1[0][0] + range1[1][1] - range1[1][0]);
				after+=area2 * (range2[0][1] - range2[0][0] + range2[1][1] - range2[1][0]);

				//if ((thisCut < bestCut) || (bestCut==-1))
				if ((before-after > bestCut))
				{
					bestCut = before-after;
					//bestCut=thisCut;
					polygonId = k;
					varId = i;
					cutId = j;
					//Strangely enough, the following produces warnings !!
					//totCut=nCut[i];
					for(q=1; q<nCut[i]; q++);
					totCut = q+1;
				}				
			}
		}
	}

	workingEnv.end();

	// Apply the best cut defined by <polygonId, varId, cutId, totCut>

	polygonSet.add(copy(env,polygonSet[polygonId]));
	IloNumArray2 range = getRangeDot(env, polygonSet[polygonId], coeff, bound);
	IloInt i,j;

	IloNumArray cutExpr(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr.add(coeff[varId][j]);
	}
	IloNum min = range[varId][0];
	IloNum max = range[varId][1];
	IloNum cut = min + (max - min) * cutId / totCut;
	cutExpr.add(cut - bound[varId]);
	polygonSet[polygonId].add(cutExpr);

	IloNumArray cutExpr2(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr2.add(-coeff[varId][j]);
	}
	cutExpr2.add(bound[varId]-cut);
	polygonSet[polygonSet.getSize()-1].add(cutExpr2);
}


// spit the polygons in <polygonSet> along the planes coeff*X+bound = 0
void splitSignPolygon(IloEnv env, PolygonSet polygonSet, IloNumArray2 coeff, IloNumArray bound)
{
	IloInt nVar=polygonSet[0][0].getSize() - 1;
	IloInt varId, k;


	for(varId=0; varId < nVar; varId++)
	{

	for(k=0; k<polygonSet.getSize(); k++)    // consider each polygon of the set
	{
		assert(nVar == polygonSet[k][0].getSize() - 1);

		IloEnv workingEnv;
		IloNumArray2 range = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
		IloNum min = range[varId][0];
		IloNum max = range[varId][1];
		displayRange(workingEnv, range);
		workingEnv.end();
		//int a; cin >> a;
	
		// if 0 is in the interior of range[varId]
		if ((min*max < 0) && (ABS(min) > EPS) && (ABS(max) > EPS))
		{
			// split the polygon
			polygonSet.add(copy(env,polygonSet[k]));
			
			// first piece: add coeff*X <= -bound
			IloInt j;
			IloNumArray cutExpr(env);
			for(j=0; j<nVar; j++)
			{
				cutExpr.add(coeff[varId][j]);
			}
			cutExpr.add(-bound[varId]);
			polygonSet[k].add(cutExpr);
			// second piece: add coeff*X >= -bound
			//		i.e. -coeff*X <= bound
			IloNumArray cutExpr2(env);
			for(j=0; j<nVar; j++)
			{
				cutExpr2.add(-coeff[varId][j]);
			}
			cutExpr2.add(bound[varId]);
			polygonSet[polygonSet.getSize()-1].add(cutExpr2);
		}
	}
	}
}


void splitPolygonFixedGridSum(IloEnv env, PolygonSet polygonSet, IloNumArray2 coeff, IloNumArray bound)
{
	IloInt nVar=polygonSet[0][0].getSize() - 1;
	IloInt k;

	IloNum bestCut=-1;
	IloInt polygonId, varId;
	IloNum cutValue;

	IloEnv workingEnv;

	for(k=0; k<polygonSet.getSize(); k++)    // consider each polygon of the set
	{
		workingEnv.out() << "**************************************************" << endl;
		workingEnv.out() << "Considering polygon #" << k << "..." << endl;
		workingEnv.out() << "**************************************************" << endl;
		assert(nVar == polygonSet[k][0].getSize() - 1);


		IloNumArray2 range = getRangeDot(workingEnv, polygonSet[k], coeff, bound);

		displayRange(workingEnv, range);

		IloInt i,j;

		for(i=0; i<nVar; i++)   // consider cutting along each variable
		{
			workingEnv.out() << "**************************************************" << endl;
			workingEnv.out() << "   Cutting along variable  #" << i << "..." << endl;
			workingEnv.out() << "**************************************************" << endl;
			IloNumArray cutExpr(workingEnv);
			IloInt q;
			for(q=0; q<nVar; q++)
			{
				cutExpr.add(coeff[i][q]);
			}
			cutExpr.add(0);
			IloNumArray cutExpr2(workingEnv);
			for(q=0; q<nVar; q++)
			{
				cutExpr2.add(-coeff[i][q]);
			}
			cutExpr2.add(0);

			IloNum min = range[i][0];
			IloNum max = range[i][1];

			workingEnv.out() << " ### Checking range for variable #" << i << endl;
			workingEnv.out() << "     Min = " << min << "     Max = " << max << endl;
			workingEnv.out() << " ### ### ### ### ### ### ### ### ###" << endl;
			//assert (nCut[i] >= 2);
			workingEnv.out() << " Range for j: [" << (int)(min/STEP)+1 << "," << (int)(max/STEP) << "]" << endl;
			j=(int)(min/STEP)+1;
			for(; (j<= (int)(max/STEP)) && (j * STEP <= max) && (j * STEP >= min) ; j++)   // consider each cut on the grid, keep the best
			{
				//workingEnv.out() << "**************************************************" << endl;
				//workingEnv.out() << "      Cutting nbr #" << j << "/" << nCut[i] << "..." << endl;
				//workingEnv.out() << "**************************************************" << endl;
				IloNum cut = j * STEP;
				workingEnv.out() << "     Cut = " << cut << endl;
				cutExpr[cutExpr.getSize()-1] = cut - bound[i];
				polygonSet[k].add(cutExpr);
	
				env.out() << endl << endl;
				displayPolygon(workingEnv, polygonSet[k]);
				
				env.out() << "Calling getRangeDot 1...";
				IloNumArray2 range1 = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
				env.out() << "End of call 1.";


				//workingEnv.out() << "min =" << min << endl;
				//workingEnv.out() << "max =" << max << endl;
				//workingEnv.out() << "j =" << j << endl;
				//displayPolygon(workingEnv, polygonSet[k]);

				IloNum area1 = getArea(workingEnv, polygonSet[k]);
				//IloNum thisCut = area1*(range1[0][0] - range1[0][1] + range1[1][0] - range1[1][1]);

				polygonSet[k].remove(polygonSet[k].getSize()-1);
				cutExpr2[cutExpr2.getSize()-1] = bound[i] - cut;
				polygonSet[k].add(cutExpr2);


				env.out() << endl << endl;
				displayPolygon(workingEnv, polygonSet[k]);
				
				env.out() << "Calling getRangeDot 2...";
				IloNumArray2 range2 = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
				env.out() << "Calling getRangeDot 2...";
				
				IloNum area2 = getArea(workingEnv, polygonSet[k]);
				//thisCut += area2*(range2[0][0] - range2[0][1] + range2[1][0] - range2[1][1]);
				polygonSet[k].remove(polygonSet[k].getSize()-1);


				//displayRange(workingEnv, range);
				//displayRange(workingEnv, range1);
				//displayRange(workingEnv, range2);
				
				IloNum before = (area1 + area2) * (range[0][1] - range[0][0] + range[1][1] - range[1][0]);
				IloNum after = area1 * (range1[0][1] - range1[0][0] + range1[1][1] - range1[1][0]);
				after+=area2 * (range2[0][1] - range2[0][0] + range2[1][1] - range2[1][0]);

				//if ((thisCut < bestCut) || (bestCut==-1))
				if ((before-after > bestCut))
				{
					bestCut = before-after;
					//bestCut=thisCut;
					polygonId = k;
					varId = i;
					cutValue = j * STEP;
				}
			}
		}
	}

	workingEnv.end();

	// Apply the best cut defined by <polygonId, varId, cutId, totCut>
	if (bestCut > 0)
	{
	polygonSet.add(copy(env,polygonSet[polygonId]));
	IloNumArray2 range = getRangeDot(env, polygonSet[polygonId], coeff, bound);
	IloInt i,j;

	IloNumArray cutExpr(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr.add(coeff[varId][j]);
	}
	IloNum min = range[varId][0];
	IloNum max = range[varId][1];
	IloNum cut = cutValue;
	cutExpr.add(cut - bound[varId]);
	polygonSet[polygonId].add(cutExpr);

	IloNumArray cutExpr2(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr2.add(-coeff[varId][j]);
	}
	cutExpr2.add(bound[varId]-cut);
	polygonSet[polygonSet.getSize()-1].add(cutExpr2);
	}
}

void splitPolygonFixedGridSum2(IloEnv env, PolygonSet polygonSet, IloNumArray2 coeff, IloNumArray bound)
{
	IloInt nVar=polygonSet[0][0].getSize() - 1;
	IloInt k;

	IloNum bestCut=-1;
	IloInt polygonId, varId;
	IloNum cutValue;

	IloEnv workingEnv;

	for(k=0; k<polygonSet.getSize(); k++)    // consider each polygon of the set
	{
		workingEnv.out() << "**************************************************" << endl;
		workingEnv.out() << "Considering polygon #" << k << "..." << endl;
		workingEnv.out() << "**************************************************" << endl;
		assert(nVar == polygonSet[k][0].getSize() - 1);


		IloNumArray2 range = getRangeDot(workingEnv, polygonSet[k], coeff, bound);

		displayRange(workingEnv, range);

		IloInt i,j;

		for(i=0; i<nVar; i++)   // consider cutting along each variable
		{
			workingEnv.out() << "**************************************************" << endl;
			workingEnv.out() << "   Cutting along variable  #" << i << "..." << endl;
			workingEnv.out() << "**************************************************" << endl;
			IloNumArray cutExpr(workingEnv);
			IloInt q;
			for(q=0; q<nVar; q++)
			{
				cutExpr.add(coeff[i][q]);
			}
			cutExpr.add(0);
			IloNumArray cutExpr2(workingEnv);
			for(q=0; q<nVar; q++)
			{
				cutExpr2.add(-coeff[i][q]);
			}
			cutExpr2.add(0);

			IloNum min = range[i][0];
			IloNum max = range[i][1];

			workingEnv.out() << " ### Checking range for variable #" << i << endl;
			workingEnv.out() << "     Min = " << min << "     Max = " << max << endl;
			workingEnv.out() << " ### ### ### ### ### ### ### ### ###" << endl;
			//assert (nCut[i] >= 2);
			workingEnv.out() << " Range for j: [" << (int)(min/STEP)+1 << "," << (int)(max/STEP) << "]" << endl;
			j=(int)(min/STEP)+1;
			for(; (j<= (int)(max/STEP)) && (j * STEP <= max) && (j * STEP >= min) ; j++)   // consider each cut on the grid, keep the best
			{
				//workingEnv.out() << "**************************************************" << endl;
				//workingEnv.out() << "      Cutting nbr #" << j << "/" << nCut[i] << "..." << endl;
				//workingEnv.out() << "**************************************************" << endl;
				IloNum cut = j * STEP;
				workingEnv.out() << "     Cut = " << cut << endl;
				cutExpr[cutExpr.getSize()-1] = cut - bound[i];
				polygonSet[k].add(cutExpr);
	
				env.out() << endl << endl;
				displayPolygon(workingEnv, polygonSet[k]);
				
				env.out() << "Calling getRangeDot 1...";
				IloNumArray2 range1 = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
				env.out() << "End of call 1.";


				//workingEnv.out() << "min =" << min << endl;
				//workingEnv.out() << "max =" << max << endl;
				//workingEnv.out() << "j =" << j << endl;
				//displayPolygon(workingEnv, polygonSet[k]);

				IloNum area1 = getArea(workingEnv, polygonSet[k]);
				//IloNum thisCut = area1*(range1[0][0] - range1[0][1] + range1[1][0] - range1[1][1]);

				polygonSet[k].remove(polygonSet[k].getSize()-1);
				cutExpr2[cutExpr2.getSize()-1] = bound[i] - cut;
				polygonSet[k].add(cutExpr2);


				env.out() << endl << endl;
				displayPolygon(workingEnv, polygonSet[k]);
				
				env.out() << "Calling getRangeDot 2...";
				IloNumArray2 range2 = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
				env.out() << "Calling getRangeDot 2...";
				
				IloNum area2 = getArea(workingEnv, polygonSet[k]);
				//thisCut += area2*(range2[0][0] - range2[0][1] + range2[1][0] - range2[1][1]);
				polygonSet[k].remove(polygonSet[k].getSize()-1);


				//displayRange(workingEnv, range);
				//displayRange(workingEnv, range1);
				//displayRange(workingEnv, range2);
				
				IloNum before = (area1 + area2) * (range[0][1] - range[0][0] + range[1][1] - range[1][0]);
				IloNum after = area1 * (range1[0][1] - range1[0][0] + range1[1][1] - range1[1][0]);
				after+=area2 * (range2[0][1] - range2[0][0] + range2[1][1] - range2[1][0]);

				//if ((thisCut < bestCut) || (bestCut==-1))
				if ((before-after > bestCut))
				{
					bestCut = before-after;
					//bestCut=thisCut;
					polygonId = k;
					varId = i;
					cutValue = j * STEP;
				}
			}
		}
	}

	workingEnv.end();

	// Apply the best cut defined by <polygonId, varId, cutId, totCut>
	if (bestCut > 0)
	{
	polygonSet.add(copy(env,polygonSet[polygonId]));
	IloNumArray2 range = getRangeDot(env, polygonSet[polygonId], coeff, bound);
	IloInt i,j;

	IloNumArray cutExpr(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr.add(coeff[varId][j]);
	}
	IloNum min = range[varId][0];
	IloNum max = range[varId][1];
	IloNum cut = cutValue;
	cutExpr.add(cut - bound[varId]);
	polygonSet[polygonId].add(cutExpr);

	IloNumArray cutExpr2(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr2.add(-coeff[varId][j]);
	}
	cutExpr2.add(bound[varId]-cut);
	polygonSet[polygonSet.getSize()-1].add(cutExpr2);
	}
}


void splitPolygonFixedGridMax(IloEnv env, PolygonSet polygonSet, IloNumArray2 coeff, IloNumArray bound)
{
	IloInt nVar=polygonSet[0][0].getSize() - 1;
	IloInt k;

	IloNum bestCut=-1;
	IloInt polygonId, varId;
	IloNum cutValue;

	IloEnv workingEnv;

	for(k=0; k<polygonSet.getSize(); k++)    // consider each polygon of the set
	{
		workingEnv.out() << "**************************************************" << endl;
		workingEnv.out() << "Considering polygon #" << k << "..." << endl;
		workingEnv.out() << "**************************************************" << endl;
		assert(nVar == polygonSet[k][0].getSize() - 1);


		IloNumArray2 range = getRangeDot(workingEnv, polygonSet[k], coeff, bound);

		displayRange(workingEnv, range);

		IloInt i,j;

		for(i=0; i<nVar; i++)   // consider cutting along each variable
		{
			workingEnv.out() << "**************************************************" << endl;
			workingEnv.out() << "   Cutting along variable  #" << i << "..." << endl;
			workingEnv.out() << "**************************************************" << endl;
			IloNumArray cutExpr(workingEnv);
			IloInt q;
			for(q=0; q<nVar; q++)
			{
				cutExpr.add(coeff[i][q]);
			}
			cutExpr.add(0);
			IloNumArray cutExpr2(workingEnv);
			for(q=0; q<nVar; q++)
			{
				cutExpr2.add(-coeff[i][q]);
			}
			cutExpr2.add(0);

			IloNum min = range[i][0];
			IloNum max = range[i][1];

			workingEnv.out() << " ### Checking range for variable #" << i << endl;
			workingEnv.out() << "     Min = " << min << "     Max = " << max << endl;
			workingEnv.out() << " ### ### ### ### ### ### ### ### ###" << endl;
			//assert (nCut[i] >= 2);
			workingEnv.out() << " Range for j: [" << (int)(min/STEP)+1 << "," << (int)(max/STEP) << "]" << endl;
			j=(int)(min/STEP)+1;
			for(; (j<= (int)(max/STEP)) && (j * STEP <= max) && (j * STEP >= min) ; j++)   // consider each cut on the grid, keep the best
			{
				//workingEnv.out() << "**************************************************" << endl;
				//workingEnv.out() << "      Cutting nbr #" << j << "/" << nCut[i] << "..." << endl;
				//workingEnv.out() << "**************************************************" << endl;
				IloNum cut = j * STEP;
				workingEnv.out() << "     Cut = " << cut << endl;
				cutExpr[cutExpr.getSize()-1] = cut - bound[i];
				polygonSet[k].add(cutExpr);
	
				env.out() << endl << endl;
				displayPolygon(workingEnv, polygonSet[k]);
				
				env.out() << "Calling getRangeDot 1...";
				IloNumArray2 range1 = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
				env.out() << "End of call 1.";


				//workingEnv.out() << "min =" << min << endl;
				//workingEnv.out() << "max =" << max << endl;
				//workingEnv.out() << "j =" << j << endl;
				//displayPolygon(workingEnv, polygonSet[k]);

				IloNum area1 = getArea(workingEnv, polygonSet[k]);
				//IloNum thisCut = area1*(range1[0][0] - range1[0][1] + range1[1][0] - range1[1][1]);

				polygonSet[k].remove(polygonSet[k].getSize()-1);
				cutExpr2[cutExpr2.getSize()-1] = bound[i] - cut;
				polygonSet[k].add(cutExpr2);


				env.out() << endl << endl;
				displayPolygon(workingEnv, polygonSet[k]);
				
				env.out() << "Calling getRangeDot 2...";
				IloNumArray2 range2 = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
				env.out() << "Calling getRangeDot 2...";
				
				IloNum area2 = getArea(workingEnv, polygonSet[k]);
				//thisCut += area2*(range2[0][0] - range2[0][1] + range2[1][0] - range2[1][1]);
				polygonSet[k].remove(polygonSet[k].getSize()-1);


				//displayRange(workingEnv, range);
				//displayRange(workingEnv, range1);
				//displayRange(workingEnv, range2);
				
				IloNum before = (area1 + area2) * (maxNum(range[0][1] - range[0][0], range[1][1] - range[1][0]));
				IloNum after = area1 * (maxNum(range1[0][1] - range1[0][0], range1[1][1] - range1[1][0]));
				after+=area2 * (maxNum(range2[0][1] - range2[0][0], range2[1][1] - range2[1][0]));

				//if ((thisCut < bestCut) || (bestCut==-1))
				if ((before-after > bestCut))
				{
					bestCut = before-after;
					//bestCut=thisCut;
					polygonId = k;
					varId = i;
					cutValue = j * STEP;
				}
			}
		}
	}

	workingEnv.end();

	// Apply the best cut defined by <polygonId, varId, cutValue>

	if (bestCut > 0)
	{

	polygonSet.add(copy(env,polygonSet[polygonId]));
	IloNumArray2 range = getRangeDot(env, polygonSet[polygonId], coeff, bound);
	IloInt i,j;

	IloNumArray cutExpr(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr.add(coeff[varId][j]);
	}
	IloNum min = range[varId][0];
	IloNum max = range[varId][1];
	IloNum cut = cutValue;
	cutExpr.add(cut - bound[varId]);
	polygonSet[polygonId].add(cutExpr);

	IloNumArray cutExpr2(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr2.add(-coeff[varId][j]);
	}
	cutExpr2.add(bound[varId]-cut);
	polygonSet[polygonSet.getSize()-1].add(cutExpr2);
	}
}


void splitPolygonFixedGridMax2(IloEnv env, PolygonSet polygonSet, IloNumArray2 coeff, IloNumArray bound)
{
	IloInt nVar=polygonSet[0][0].getSize() - 1;

	IloNumArray2 range;
	IloInt polygonId;
	IloEnv workingEnv;

	// First, find the polygon with maximal badness
	
	IloNum badness=0;
	for(IloInt k=0; k<polygonSet.getSize(); k++)    // consider each polygon of the set
	{
		//workingEnv.out() << "**************************************************" << endl;
		//workingEnv.out() << "Considering polygon #" << k << "..." << endl;
		//workingEnv.out() << "**************************************************" << endl;
		assert(nVar == polygonSet[k][0].getSize() - 1);

		range = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
		//displayRange(workingEnv, range);
		IloNum currentBadness = getArea(workingEnv, polygonSet[k]) * (maxNum(range[0][1] - range[0][0], range[1][1] - range[1][0]));
		if (currentBadness > badness) {polygonId=k; badness = currentBadness; }
	}
	
	// Second, find the best cut 	
	IloNum bestCut=-1;
	IloInt varId;
	IloNum cutValue;

	IloInt i,j;
	range = getRangeDot(workingEnv, polygonSet[polygonId], coeff, bound);
	for(i=0; i<nVar; i++)   // consider cutting along each variable
	{
		//workingEnv.out() << "**************************************************" << endl;
		//workingEnv.out() << "   Cutting along variable  #" << i << "..." << endl;
		//workingEnv.out() << "**************************************************" << endl;
		IloNumArray cutExpr(workingEnv);
		IloInt q;
		for(q=0; q<nVar; q++)
		{
			cutExpr.add(coeff[i][q]);
		}
		cutExpr.add(0);
		IloNumArray cutExpr2(workingEnv);
		for(q=0; q<nVar; q++)
		{
			cutExpr2.add(-coeff[i][q]);
		}
		cutExpr2.add(0);

		IloNum min = range[i][0];
		IloNum max = range[i][1];

		//workingEnv.out() << " ### Checking range for variable #" << i << endl;
		//workingEnv.out() << "     Min = " << min << "     Max = " << max << endl;
		//workingEnv.out() << " ### ### ### ### ### ### ### ### ###" << endl;
		//assert (nCut[i] >= 2);
		workingEnv.out() << " Range for j: [" << (int)(min/STEP)+1 << "," << (int)(max/STEP) << "]" << endl;
		j=(int)(min/STEP)+1;
		for(; (j<= (int)(max/STEP)) && (j * STEP <= max) && (j * STEP >= min) ; j++)   // consider each cut on the grid, keep the best
		{
			//workingEnv.out() << "**************************************************" << endl;
			//workingEnv.out() << "      Cutting nbr #" << j << "/" << nCut[i] << "..." << endl;
			//workingEnv.out() << "**************************************************" << endl;
			IloNum cut = j * STEP;
			workingEnv.out() << "     Cut = " << cut << endl;
			cutExpr[cutExpr.getSize()-1] = cut - bound[i];
			polygonSet[polygonId].add(cutExpr);

			env.out() << endl << endl;
			displayPolygon(workingEnv, polygonSet[polygonId]);
				
			//env.out() << "Calling getRangeDot 1...";
			IloNumArray2 range1 = getRangeDot(workingEnv, polygonSet[polygonId], coeff, bound);
			//env.out() << "End of call 1.";


			//workingEnv.out() << "min =" << min << endl;
			//workingEnv.out() << "max =" << max << endl;
			//workingEnv.out() << "j =" << j << endl;
			//displayPolygon(workingEnv, polygonSet[polygonId]);

			IloNum area1 = getArea(workingEnv, polygonSet[polygonId]);
			//IloNum thisCut = area1*(range1[0][0] - range1[0][1] + range1[1][0] - range1[1][1]);

			polygonSet[polygonId].remove(polygonSet[polygonId].getSize()-1);
			cutExpr2[cutExpr2.getSize()-1] = bound[i] - cut;
			polygonSet[polygonId].add(cutExpr2);


			//env.out() << endl << endl;
			//displayPolygon(workingEnv, polygonSet[polygonId]);
				
			//env.out() << "Calling getRangeDot 2...";
			IloNumArray2 range2 = getRangeDot(workingEnv, polygonSet[polygonId], coeff, bound);
			//env.out() << "Calling getRangeDot 2...";
				
			IloNum area2 = getArea(workingEnv, polygonSet[polygonId]);
			//thisCut += area2*(range2[0][0] - range2[0][1] + range2[1][0] - range2[1][1]);
			polygonSet[polygonId].remove(polygonSet[polygonId].getSize()-1);


			//displayRange(workingEnv, range);
			//displayRange(workingEnv, range1);
			//displayRange(workingEnv, range2);
				
			
			IloNum currentCut = area1 * (maxNum(range1[0][1] - range1[0][0], range1[1][1] - range1[1][0]));
			currentCut+=area2 * (maxNum(range2[0][1] - range2[0][0], range2[1][1] - range2[1][0]));

			if ((currentCut < bestCut) || (bestCut==-1))
			{
				bestCut = currentCut;
				varId = i;
				cutValue = j * STEP;
			}
		}
	}


	workingEnv.end();

	// Apply the best cut defined by <polygonId, varId, cutValue>
	if (bestCut >= 0)
	{

	polygonSet.add(copy(env,polygonSet[polygonId]));
	IloNumArray2 range = getRangeDot(env, polygonSet[polygonId], coeff, bound);
	//IloInt i,j;

	IloNumArray cutExpr(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr.add(coeff[varId][j]);
	}
	IloNum min = range[varId][0];
	IloNum max = range[varId][1];
	IloNum cut = cutValue;
	cutExpr.add(cut - bound[varId]);
	polygonSet[polygonId].add(cutExpr);

	IloNumArray cutExpr2(env);
	for(j=0; j<nVar; j++)
	{
		cutExpr2.add(-coeff[varId][j]);
	}
	cutExpr2.add(bound[varId]-cut);
	polygonSet[polygonSet.getSize()-1].add(cutExpr2);
	}
}



void splitPolygonHalfRange(IloEnv env, PolygonSet polygonSet, IloNumArray2 coeff, IloNumArray bound)
{
	IloInt nVar=polygonSet[0][0].getSize() - 1;

	IloNum bestCut=-1;
	IloInt polygonId, varId;

	IloEnv workingEnv;

	for(IloInt k=0; k<polygonSet.getSize(); k++)    // consider each polygon of the set
	{
		workingEnv.out() << "**************************************************" << endl;
		workingEnv.out() << "Considering polygon #" << k << "..." << endl;
		workingEnv.out() << "**************************************************" << endl;
		assert(nVar == polygonSet[k][0].getSize() - 1);

		IloNumArray2 range = getRangeDot(workingEnv, polygonSet[k], coeff, bound);
		
		for(IloInt i=0; i<nVar; i++)
		{
			if (range[i][1] - range[i][0] > bestCut)
			{
				bestCut = range[i][1] - range[i][0];
				polygonId=k;
				varId=i;
			}
		}
		//displayRange(workingEnv, range);
	}

	workingEnv.end();

	


	//env.out() << "Max range size: " << bestCut << endl;
	//int a; cin >> a;

	if (bestCut > 1.6 * STEP)
	{
		// Apply the best cut defined by <polygonId, varId>

		polygonSet.add(copy(env,polygonSet[polygonId]));
		IloNumArray2 range = getRangeDot(env, polygonSet[polygonId], coeff, bound);
		IloInt i,j;

		IloNumArray cutExpr(env);
		for(j=0; j<nVar; j++)
		{
			cutExpr.add(coeff[varId][j]);
		}
		IloNum min = range[varId][0];
		IloNum max = range[varId][1];
		IloNum cut = min + (max-min)/2;
		env.out() << cut << endl;
		cut =((int) (cut/STEP)) * STEP ;
		if (cut < min) cut = cut+STEP;
		if ((cut<min) || (cut >max)) {env.out() << "Problem during the cut !!" << endl; int a; cin >> a; }
		//env.out() << cut << endl;
		//int a; cin >> a;

		cutExpr.add(cut - bound[varId]);
		polygonSet[polygonId].add(cutExpr);

		IloNumArray cutExpr2(env);
		for(j=0; j<nVar; j++)
		{
			cutExpr2.add(-coeff[varId][j]);
		}
		cutExpr2.add(bound[varId]-cut);
		polygonSet[polygonSet.getSize()-1].add(cutExpr2);
	}
}


void refineByReachability(IloEnv env, PolygonSet polygonSet1, IloNumArray2 coeff1, IloNumArray bound1, PolygonSet polygonSet2, IloNumArray2 coeff2, IloNumArray bound2)
{
	RatPolygonSet ratPolygonSet1(env);
	for(IloInt i=0; i<polygonSet1.getSize(); i++)
	{
		ratPolygonSet1.add(rationalizePolygon(env, polygonSet1[i], UPPER));
	}
	RatPolygonSet ratPolygonSet2(env);
	for(IloInt i=0; i<polygonSet2.getSize(); i++)
	{
		ratPolygonSet2.add(rationalizePolygon(env, polygonSet2[i], UPPER));
	}

	char name[10]  = "prelim";


	displayHytechLocation(env, ratPolygonSet1, coeff1, bound1, name, 1, polygonSet1);
	displayHytechLocation(env, ratPolygonSet2, coeff2, bound2, name, 2, polygonSet2);


	char cmd[75 + 4*strlen(name)];
	strcpy(cmd, "cat heatBegin.hy ");
	strcat(cmd, name); strcat(cmd, "1.hy ");
	strcat(cmd, name); strcat(cmd, "2.hy heatEnd.hy "); 
	strcat(cmd, name); strcat(cmd, "Print1.hy "); 
	strcat(cmd, name); strcat(cmd, "Print2.hy > prelim.hy"); 
	system(cmd);

	system("hytech prelim.hy > /tmp/log 2> /tmp/errlog");

	system("parsing/parserHeater /tmp/log > /tmp/reachPolygons.txt");

	system("sed -n -e \"1,/BeginNext1/ p\" /tmp/reachPolygons.txt | sed -e \"s/BeginNext1//\" > /tmp/reachSetPrelim1.txt");
	system("sed -n -e \"/BeginNext1/,/EndNext1/ p\" /tmp/reachPolygons.txt | sed -e \"s/BeginNext1//\" | sed -e \"s/EndNext1//\" > /tmp/reachSetPrelim2.txt");

			
	ratPolygonSet1.end();
	ratPolygonSet2.end();

	while(polygonSet1.getSize() > 0)
	{
		polygonSet1[polygonSet1.getSize() - 1].end();
		polygonSet1.remove(polygonSet1.getSize() - 1);
	}
	while(polygonSet2.getSize() > 0)
	{
		polygonSet2[polygonSet2.getSize() - 1].end();
		polygonSet2.remove(polygonSet2.getSize() - 1);
	}

	//polygonSet1.end();
	//polygonSet2.end();

	ifstream f("/tmp/reachSetPrelim1.txt");
	f >> polygonSet1;
	f.close();

	f.open("/tmp/reachSetPrelim2.txt");
	f >> polygonSet2;
	f.close();

}

*/




