#ifndef GUARD_automaton_h
#define GUARD_automaton_h
/***************************************************************************
                          automaton.h  -  description
                             -------------------
    begin                : Thu Nov 21 2002
    copyright            : (C) 2002 by Goran Frehse
    email                : goran@s.cs.auc.dk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

//#define DEBUG 1

#include <stdio.h>
#include <iostream>
#include <fstream>
#include <vector>
#include <list>
#include <set>
#include <map>
#include <algorithm>
#include <stdexcept>

//#include "parameters.h"
#include "general.h"
#include "clock_val_set.h"
#include "symb_states.h"
#include "stopwatch.h"
#include "location.h"
#include "transition.h"
#include "deriv_ops.h" // for refinement
#include "rat_linexpression.h"
#include "probabilistic_alternative.h"

// for optimal partitioning

using namespace Parma_Polyhedra_Library;
using namespace Parma_Polyhedra_Library::IO_Operators;
using namespace std;

//extern unsigned int VERBOSE_LEVEL;
extern unsigned int DEBUG_OUTPUT;

// -------------------------------------------------
// Parameters
// -------------------------------------------------
extern int TIME_POST_ITER;
extern unsigned TIME_POST_CONE_ITER;
extern Rational TIME_POST_LAMBDA;

extern int CONSTRAINT_BITSIZE;
extern bool ELAPSE_TIME; // perform time elapse (can be turned off for discrete time systems)
extern bool BLOCK_OVERAPPROXIMATION;
// Composition
extern bool STOP_COMPOSE_AT_ERROR;
extern string ERROR_LOCATION_STRING; 
extern string TAU_LABEL_STRING; 
// Reachability
extern bool CHEAP_CONTAIN_RETURN_OTHERS;
extern bool USE_CONVEX_HULL; // this is for all reach operations, is temporarily changed in is_simulation()
extern bool REACH_STOP_AT_FORB; // check after each iteration and stop reachability if forbidden states encountered
extern bool REACH_ONLY_EXPLORE; // do not keep visited states, explore only
extern bool REACH_USE_BBOX;
extern bool USE_CONSTRAINT_HULL;
extern int REACH_USE_BBOX_ITER;
extern int REACH_STOP_USE_CONVEX_HULL_ITER;
extern int REACH_STOP_USE_BITSIZE;
extern bool REACH_STOP_USE_CONVEX_HULL_SETTLE;
extern int REACH_MAX_ITER;
extern int REACH_CONSTRAINT_LIMIT;
extern int TP_CONSTRAINT_LIMIT;
extern int REACH_CONSTRAINT_TRIGGER;
extern int REACH_BITSIZE_TRIGGER;
extern int SEARCH_METHOD;
extern int GLOBAL_ITER_COUNT;
extern int SNAPSHOT_INTERVAL;
extern bool DEADLOCK_CHECKING;
// Simulation
extern bool Q_IS_DETERMINISTIC; // this doesn't work, leave it off
extern bool CHECK_FOR_TAU_IN_Q; // if tau transitions are followed in Q
extern bool USE_CONVEX_HULL_FOR_PRIMING; // this is for priming the simulation relation
extern bool STOP_AT_BAD_STATES; // can't be used with USA_CONVEX_HULL_FOR_PRIMING
extern bool PRIME_R_WITH_REACH; // prime R with reach set of P||Q
extern bool PRIME_R_WITH_DISCRETE_REACH;
extern bool SHOW_BAD_STATES; // output bad states where found
extern bool SIM_SIMPLIFY_R; // simplify R after difference operation
// Minimization
extern bool MINIM_USE_CONVEX_HULL_FOR_PRIMING; // this is for priming the simulation relation in minimization
extern bool MINIM_PRIME_R_WITH_REACH; // prime R with reach set of P||Q in minimization
// Composition
extern bool COMPOSE_WITH_REACH_MIN; // delete unreachable locations after composition
extern bool COMPOSE_USE_CONVEX_HULL_FOR_REACH; // this is for priming the simulation relation in minimization
extern bool USE_HIOA_AUTOMATA;

// For transitions that aren't allowed to be removed
extern trans_ref_set DONT_REMOVE_TRANS;

// -------------------------------------------------
// For refinement 
// -------------------------------------------------
extern int CONSTRAINT_BITSIZE;
// Refinement parameters from parameters.h
extern bool REFINE_CHECK_TIME_RELEVANCE;
extern bool REFINE_CHECK_TIME_RELEVANCE_DURING; // check the time relevance of existing transitions for cells during the refinement process
extern bool REFINE_CHECK_TIME_RELEVANCE_FINAL; // check the time relevance of existing transitions for cells that are completely refined
extern bool REFINE_CHECK_TRANS_DIMS;
extern int  REFINE_DERIVATIVE_METHOD;
extern bool REFINE_TEST_REACH_SPLIT;
extern bool REFINE_PRIORITIZE_REACH_SPLIT;
extern bool REFINE_PRIORITIZE_ANGLE;
extern bool REFINE_SMALLEST_FIRST;
extern bool REFINE_USE_FP;
extern double REFINE_DERIV_MINANGLE;
extern int REFINE_PARTITION_LEVEL_MAX;
extern int REFINE_PARTITION_LEVEL_DELTA;
extern bool REFINE_PARTITION_INSIDE;
extern int REACH_FB_REFINE_METHOD;
extern int REFINE_INTERSECT_METHOD;
extern bool REFINE_FORCE_SPLITTING;
extern int REFINE_MAX_CHECKS;

typedef list <location_ref> loc_ref_list;
template <class T>
std::ostream& operator<<( std::ostream& os, const list <T> &c )
    {
      os << "[";
      for (typename list<T>::const_iterator j=c.begin();j!=c.end();++j)
      {
        if (j!=c.begin())
          os << "," << *j;
        else
          os << *j;
      };
      os << "]";
      return os;
    }

enum refinement_method {carth_center,carth0_center,carth1_center,deriv_center,jacob_center,deriv_dcenter,jacob_dcenter};

// global list for refinement constraints
extern list <ConstraintRatPair> REFINE_CONSTRAINTRATPAIRLIST;
extern label_ref REFINEMENT_LABEL;


class automaton
{
  public:
  automaton();
  automaton(string nme);
  automaton(clock_dim_type d);
  
  variable_id_map get_var_names() const;
  variable_id_map get_var_names_mu() const;
  vector <string> get_loc_names() const;
	loc_names_map	 get_loc_names_map() const;
  string get_label_name(label_ref l) const;
  label_ref get_label_ref(string label_name);
	symb_states_type get_ini_states() const;
	symb_states_type get_invariants() const;
   int get_memory() const;

	map<label_ref, string> get_label_names_map() const;
   
  // Manipulation Methods
  void clear();
  
//  location_ref add_location(AUT_invariant invar, string sname, const Constraint_List& post_poly);
  location_ref add_location(AUT_invariant invar, string sname, Constraint_List post_poly);
	location_ref add_location(string sname, location l);
  void assume_loc_modification(location_ref loc);
	void location_remove_nonface_silents(location_ref loc,label_ref splitting_silent_label);
  void location_remove_nontimerel_silents(location_ref loc,label_ref splitting_silent_label);
  void transition_remove_refs(transition_ref t);

  transition_ref add_transition(location_ref sl, label_ref a, clock_val_set mu, location_ref tl, bool is_urgent=false, distribution_ref distr = 0, unsigned int distr_entry = 0);
  transition_ref add_transition(string source_name, string label_name, clock_val_set mu, string target_name, distribution_ref distr = 0, unsigned int distr_entry = 0);
  transition_ref add_transition(string source_name, string label_name, clock_val_set guard, clock_val_set mudest, string target_name, distribution_ref distr = 0, unsigned int distr_entry = 0);

  transition_ref add_transition(const location_ref& sl, const label_ref& a, const AUT_guard& g, const clock_ref_set& r, const location_ref& tl);
  transition_ref add_transition(string source_name, string label_name, const AUT_guard& g, const clock_ref_set& r, string target_name);

  void add_transitions_from_distribution(string source_name, string label_name, clock_val_set guard, const list<prob_alt> &palist);
  void add_distribution(const distribution& dist);
	
	clock_val_set get_restricted_mu(transition_ref t) const;
	clock_val_set get_restricted_entry_set(transition_ref t) const;
	clock_val_set get_restricted_exit_set(transition_ref t) const;
  
   void ini_states_assign(symb_states_type& states);
  
   void add_space_dimensions_and_embed(dimension_type ndims);
   void map_space_dimensions(const PFunction& pfunc);
   void project_to_vars(const clock_ref_set& crs);

  void invariant_assign(const symb_states_type& states, bool map_locs=false);
  void invariant_assign_exec(const symb_states_type& states);

  clock_ref add_variable(string var_name);
  clock_ref add_ext_variable(string var_name);
  clock_ref add_parameter(string var_name);

  label_ref add_label(string label_name);
  
//  void expand_labels(label_ref_set labs);
  void expand_labels(label_ref_set labs, map <string,label_ref> lmap);

  void replace_labels(label_ref_set labs, label_ref l);
  void replace_labels(map<label_ref,label_ref> label_map);
	
	void reverse();

	bool transition_is_active(transition_ref& tr) const;
	int active_transitions_size() const;
	
  void print_labels() const;
  void print_size() const;
  void print() const;
  void print_phaver(ostream& o) const;  
  void print_dot(ostream& o) const;  
  void print_inv_fp_raw(ostream& o) const;  
  void print_surface_fp_raw(ostream& o) const;  
  void print_snapshot_fp_raw(const symb_state_maplist& states, string filename) const;
	void print_graph(ostream& o, clock_ref_set& pvars, symb_states_type& reach) const;
  // ---------------------------------------------------------------------------
  // Post and pre operations
	
	void clear_priority(symb_state_maplist& states);
	
	void assign_priority(symb_state_maplist& states, const location_ref& loc, int& time, bool reachable_only=false);
	
	void assign_priority(symb_state_maplist& states, const symb_state_plist& new_states, bool reachable_only=false);
	
	bool get_strongly_connected_locs(symb_state_maplist& states, location_ref& loc, int hor, loc_ref_set& lrs);
	
	void add_post_locs(symb_state_maplist& states, int hor, loc_ref_set& lrs, bool reachable_only);

	void get_post_locs(symb_state_maplist& states, location_ref& loc, int hor, loc_ref_set& lrs, bool reachable_only=false);

	void get_pre_locs(symb_state_maplist& states, location_ref& loc, int hor, loc_ref_set& lrs, bool reachable_only=false);
	
	double loc_distance(symb_state_maplist& states, location_ref& source_loc, location_ref& target_loc);
	
	double get_active_post_loc_ratio(symb_state_maplist& states, location_ref& loc, int hor, const symb_state_plist& new_states);
	
	double get_active_pre_loc_ratio(symb_state_maplist& states, location_ref& loc, int hor, const symb_state_plist& new_states);
	
  //symb_state_maplist::iterator pop_maplist_iter(symb_state_plist& new_states);
	symb_state_maplist::iterator pop_maplist_iter(symb_state_maplist& states, symb_state_plist& new_states, location_ref& last_loc);
	
  void insert_maplist_iter(symb_state_maplist& states, symb_state_plist& new_states, const symb_state_maplist::iterator& new_iter);

  void print(symb_state_maplist& m, automaton& aut);
	
	void simplify_successor(location_ref& loc, clock_val_set& cvs);

  void time_post_assign_convex(location_ref& loc, clock_val_set& cvs);

  void time_post_assign(clock_val_set& w, const clock_val_set& inv, const time_elapse_poly& tp);

  void time_post_assign(location_ref& loc, clock_val_set& cvs);

  void time_post_assign(symb_state_maplist& states, symb_state_plist& new_states);

  symb_states_type trans_and_time_post(const symb_states_type& states);
  
  void trans_and_time_post_assign(symb_state_maplist& states, symb_state_plist& new_states, symb_states_type& f_states, bool& f_states_reachable, label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true);
	
  //bool check_partitioning(symb_state_maplist& states, symb_state_plist& check_states, const location_ref& loc, const int iter_count, bool checktime);
  bool check_partitioning(symb_state_maplist& states, list< symb_state >::iterator& state_it, symb_state_plist& check_states, const location_ref& loc, const int iter_count, bool checktime);

	
	void reach_init(symb_states_type i_states, symb_state_maplist& states, symb_state_plist& new_states, bool checktime=true);

  symb_states_type get_reach_set(const symb_states_type& i_states, symb_states_type& f_states, bool& f_states_reachable, const symb_states_type& unsafe_states, string graph_file="", label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true);

  symb_states_type get_reach_set(const symb_states_type& ini_states, symb_states_type& target_states, bool& is_reachable, label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true);
  
  symb_states_type get_reach_set(label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true);
  
  symb_states_type get_reach_set_forwarditer(int delta=1, label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true);
  
	bool is_reachable(symb_states_type& orig_target, symb_states_type& reach_set, label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true);
	bool is_reachable_fb(symb_states_type& orig_target, symb_states_type& reach_set, label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true);
	
  // ---------------------------------------------------------------------------
  // Parametric Reachability
  // new 10/2003
  // ---------------------------------------------------------------------------

  clock_val_set get_param_reach(symb_states_type& forb_states, clock_ref_set params, bool stop_iter=false, bool prime_convex_hull=false);
  
  // ---------------------------------------------------------------------------
  // Simulation Relation
  // new 08/2003
  // ---------------------------------------------------------------------------

  // Transition-Conform States

  // ---------- here mu is presumed not to be intersected with the invariants ---->
  clock_val_set sim_states_EP_OLD(const transition_ref& tr) const;
  clock_val_set sim_states_EQ_OLD(const transition_ref& tr, const clock_val_set& R, const dimension_type& pdim) const;
  // <-----------------------------------

  clock_val_set sim_states_EP(const transition_ref& tr) const;
  clock_val_set sim_states_EQ(const transition_ref& tr, const clock_val_set& R, const dimension_type& pdim) const;

  clock_val_set sim_states_Btr(const transition_ref& s, const location_ref_pair& lrp, state_relation& R, const automaton& spec_aut);
  
  // Time-Conform States

  clock_val_set elapse_succ_set(clock_val_set cvs, time_elapse_poly tp, clock_val_set inv);

  clock_val_set elapse_succ_set_from(dimension_type dimm, clock_val_set cvs, time_elapse_poly tp, clock_val_set inv);

  clock_val_set good_elapse_states_in_Q(const location_ref& k, const location_ref& l, state_relation& R, const automaton& spec_aut);

  clock_val_set sim_states_Bte(const location_ref& k, const location_ref& l, state_relation& R, const automaton& spec_aut, const clock_val_set& equiv_cvs);

  void void_sim_rel(state_relation& R);

  bool get_sim_rel(state_relation& R, automaton& spec_aut, const clock_val_set& equiv_cvs, bool get_ident_bisim=false, label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true);

  void get_bisim_rel(state_relation& R, automaton& spec_aut, const clock_val_set& equiv_cvs);
  
  void reach_discrete_assign(state_relation& R, const clock_val_set& cvs_equiv, const automaton& spec_aut);
  
  void reach_assign(state_relation& R, const clock_val_set& cvs_equiv, automaton& spec_aut, label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true, bool discrete_reach=false);

  void restrict_to_invariant(symb_states_type& states) const;

  state_relation prime_R(const clock_val_set& cvs_equiv, automaton& spec_aut, bool get_bisimimulation=false, label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true);

  bool check_ini_in_rel(const clock_val_set& cvs_equiv, automaton& spec_aut, state_relation& R);

  bool is_simulation(const clock_val_set& cvs_equiv, automaton& spec_aut, bool get_bisimimulation=false, label_ref_set dontchecklabs=empty_label_ref_set(), bool checktime=true, state_relation* pR=NULL);

  bool is_simulation(automaton& spec_aut, state_relation* pR=NULL, bool get_bisimimulation=false);

  void map_or_add_labels(map<string,label_ref> &label_map);

  // ---------------------------------------------------------------------------
  // Composition
  // ---------------------------------------------------------------------------

  bool add_location_from_composition(location_ref_to_location_ref_pair_map& loc_map, const automaton& aut1, const location_ref& loc1, const automaton& aut2, const location_ref& loc2, const PFunction& pfunc1, const PFunction& pfunc2);

  void add_transition_from_composition(location_ref& loc1, location_ref& loc2, const automaton& aut1, const transition_ref& t1, const map <label_ref,label_ref>& label1_to_common_label, const automaton& aut2, const transition_ref& t2, const map <label_ref,label_ref>& label2_to_common_label, const PFunction& pfunc1, const PFunction& pfunc2, distribution_ref distr = 0, unsigned int distr_entry = 0);

  void add_indep_transition_from_composition(location_ref& loc1, location_ref& loc2, const automaton& aut1, const transition_ref& t1, const map <label_ref,label_ref>& label_to_common_label, const automaton& aut2, const PFunction& pfunc1, const PFunction& pfunc2, distribution_ref distr = 0, unsigned int distr_entry = 0);

  symb_states_type minimize_reachable();

  //location_ref_to_location_ref_pair_map compose_discrete(automaton& caut, automaton spec_aut );

  //location_ref_to_location_ref_pair_map compose_discrete_intersect(automaton& caut, automaton& spec_aut );

  void get_disabled_states(state_relation& R, const automaton& spec_aut, automaton& caut, label_ref_set& contr_labels, dimension_type d_dim, location_ref_to_location_ref_pair_map& lmap);

  void restrict_mu_to_invariants(transition_ref& i);

  void restrict_mu_to_invariants_loc(location_ref loc);
  
  void restrict_mu_to_invariants();

  void add_environment_transitions();

  void assign_tts();

  // -------------------------------------------------------------------------------------------------------------------------
  //  Minimization
  // -------------------------------------------------------------------------------------------------------------------------
  bool check_consistency();

  void erase_transition(transition_ref k);

  void erase_location(location_ref l);

  symb_states_type reach_assign_symmetric(state_relation& R, const clock_val_set& cvs_equiv);

  symb_states_type minimize_locsim(const clock_val_set& cvs_equiv, bool use_bisimulation=false);

  //
  clock_val_set get_union_of_exit_sets(const location_ref loc, const label_ref lab);

  clock_val_set get_cvs_equiv(automaton& spec_aut);
	
  // -------------------------------------------------------------------------------------------------------------------------
	// Refinement and Partitioning
  // -------------------------------------------------------------------------------------------------------------------------
  
	bool location_is_surface(location_ref k, const symb_states_type& ss);
	bool location_is_surface(location_ref k, const symb_state_maplist& ss);
	int add_surface_locations(const  symb_states_type& ss, loc_ref_set& lrs);
	int unlock_surface_locations(const symb_states_type& ss);
	int unlock_locations();
	int set_surface_flag(bool val);

	bool is_fully_refined(const loc_ref_set& lrs);

	transition_ref copy_transition_with_source(transition_ref t, location_ref source, clock_val_set mu);
	transition_ref copy_transition_with_target(transition_ref t, location_ref target, clock_val_set mu);
	void copy_transition_with_self_loop(transition_ref t, location_ref new_loc, location_ref loc, clock_val_set mu);
  // -------------------------------------------------------------------------------------------------------------------------
  // -------------------------------------------------------------------------------------------------------------------------
  // Properties
  // -------------------------------------------------------------------------------------------------------------------------
  // -------------------------------------------------------------------------------------------------------------------------
  vector<location>    locations;
  vector<transition>  transitions;
  clock_dim_type      dim;
  string              name;
  symb_states_type    ini_states;
  label_ref_set       labels;
  clock_ref_set       variables; // controlled variables; uncontrolled (input) = {0,...,dim} \setminus variables
  clock_ref_set       parameters;
  clock_ref_set       input_variables;
  bool                reversed;
  int                 priority;
  map <string,location_ref>   loc_name_to_loc_ref_map;
  map <string,label_ref>      label_name_to_label_ref_map;
   variable_id_map var_id_map;

  vector<distribution> distributions;
};

PFunction get_PFunction(automaton& P1,automaton& P1Q2,bool fill_up_to=true);

// Location Methods
  
void get_loc_vert(automaton& aut,location_ref loc,double_point_list& pl);

void get_loc_deriv(automaton& aut,location_ref loc,double_point_list& vert_list,double_point_list& deriv_list);
void get_loc_deriv_nonfp(automaton& aut,location_ref loc,const clock_val_set& restr,double_point_list& deriv_list);

Constraint_List get_deriv_over_states(automaton& aut,location_ref loc);
  
void set_loc_deriv(automaton& aut,location_ref loc,double_point_list& deriv_list);
 
convex_clock_val_set get_loc_deriv(automaton& aut,location_ref loc);

convex_clock_val_set get_loc_deriv(automaton& aut,location_ref loc,clock_val_set restr);

void refine_loc_deriv(automaton& aut,location_ref loc);

void refine_loc_deriv(automaton& aut,loc_ref_list& locs);
 
void refine_loc_deriv(automaton& aut);

double get_loc_angle(automaton& aut, location_ref loc);

double get_loc_angle(automaton& aut, location_ref loc, clock_val_set& restr);

// Methods for splitting locations

void split_location(automaton& aut,label_ref splitting_silent_label,location_ref loc,Constraint c, loc_ref_list& new_locs);

void split_location(automaton& aut,label_ref splitting_silent_label,location_ref loc,clock_val_set& inv1, clock_val_set& inv2, loc_ref_list& new_locs);

void split_location(automaton& aut,label_ref splitting_silent_label,const Constraint_List& cl, loc_ref_list& locs);

// Methods for refining locations

void get_splitting_constraints(automaton& aut, location_ref loc, Constraint_List& cl, refinement_method method=carth_center);

void refine_locs(automaton& aut,label_ref splitting_silent_label, loc_ref_list& locs, refinement_method method=carth_center);

void refine_locs(automaton& aut,label_ref splitting_silent_label, refinement_method method=carth_center, int iter=1);

//typedef pair<int,Rational> refine_criterion;
typedef list<double> refine_criterion;

struct refine_criterion_comp
{
  bool operator()(const pair<refine_criterion,Constraint> s1, const pair<refine_criterion,Constraint> s2) const
  {
    return (s1.first < s2.first);
  //return __x.first < __y.first || (!(__y.first < __x.first) && __x.second < __y.second);
//    return (s1.first.first < s2.first.first || !(s2.first.first < s1.first.first) && (s1.first.second < s2.first.second));
  }
};

#include "optimal_split.h"

bool get_refinement_constraint(automaton& aut,location_ref loc,const clock_val_set& inv,const clock_val_set& reached,Constraint& con,bool& splits_reached);

bool get_refinement_invariants(automaton& aut,location_ref loc,const clock_val_set& inv,const clock_val_set& reached,clock_val_set& inv1,clock_val_set& inv2,bool& splits_reached);

bool refine_location_otf(automaton& aut,location_ref tloc,clock_val_set& reached,symb_state_maplist& states,symb_state_plist& check_states,symb_state_plist& new_states,loc_ref_set& succ_locs, bool convex);

void refine_states(automaton& aut,symb_state_maplist& states,symb_state_plist& check_states);

void refine_states(automaton& aut,symb_states_type& s);

string get_composed_aut_name(const string& n1, const string& n2);
void get_common_labels(const map <string,label_ref>& lrm1, const label_ref_set& l1, const map <string,label_ref>& lrm2, const label_ref_set& l2, map <string,label_ref>& lrm_res, label_ref_set& l_res, map <label_ref,label_ref>& label1_to_label2, map <label_ref,label_ref>& label2_to_label1, map <label_ref,label_ref>& label1_to_clabels, map <label_ref,label_ref>& label2_to_clabels);
location_ref_to_location_ref_pair_map compose_discrete(automaton& caut, const automaton& aut, const automaton& spec_aut );

bool
get_or_add_location(automaton& caut, const vector<automaton>& aut, map <loc_ref_vec, location_ref>& glmap, const loc_ref_vec& lvec_ref, location_ref& loc);

label_ref_set get_outgoing_labels(const automaton& aut, const location_ref& loc);
label_ref_set get_nonblocking_labels(const automaton& aut);

location_ref_to_location_ref_pair_map
compose_by_vector(automaton& caut, vector<automaton>& aut);

void communication_graph(const vector<automaton>& aut, ostream& o, bool do_discr=true, bool do_cont=true);
 
template <typename T>
class iterator_vector
{
   public:
   iterator_vector<T>(const vector<T>& bg, const vector<T>& en) : vec_it_begin(bg),vec_it_end(en)
   {
      vec_it=bg; // initialize with beginning
   }
   
   T& operator[](uint i)
   {
      return vec_it[i];
   }
   
   bool operator!=(const vector<T>& v)
   { return vec_it!=v; }
   
   bool operator==(const vector<T>& v)
   { return vec_it==v; }
   
   uint incr_iter_to_end(int iter_aut)
   {
      // increase this level and reset all following
      // this version increases up to the end iterator
      uint last_inc_i=0;
      bool increased_succ=false;
      while(iter_aut>=0 && !increased_succ && (vec_it!=vec_it_end))
      {
         if (vec_it[iter_aut]==vec_it_end[iter_aut])
         {
            --iter_aut;
         }
         else
         {  // increasable
            ++vec_it[iter_aut];
            last_inc_i=iter_aut;
            // reset all the rest
            for (uint j=iter_aut+1;j<vec_it.size();++j)
               vec_it[j]=vec_it_begin[j];
            increased_succ=true;
            iter_aut=vec_it.size()-1;
         }
      }      
      if (!increased_succ)
         vec_it=vec_it_end;
         
      return last_inc_i;
   }
   
   uint incr_iter_to_end()
   {
      return incr_iter_to_end(vec_it.size()-1);
   }
   
   uint incr_iter(int iter_aut)
   {
      // increase this level and reset all following
      // this version does not go up to the end iterator, thus always provides
      // either a dereferenceable iterator or the end iterator
      uint last_inc_i=0;
      bool increased_succ=false;
      while(iter_aut>=0 && !increased_succ && (vec_it!=vec_it_end))
      {
         if (vec_it[iter_aut]==vec_it_end[iter_aut] || ++vec_it[iter_aut]==vec_it_end[iter_aut])
         {
            --iter_aut;
         }
         else
         {  // increasable
//            ++vec_it[iter_aut];
            last_inc_i=iter_aut;
            // reset all the rest
            for (uint j=iter_aut+1;j<vec_it.size();++j)
               vec_it[j]=vec_it_begin[j];
            increased_succ=true;
            iter_aut=vec_it.size()-1;
         }
      }      
      if (!increased_succ)
         vec_it=vec_it_end;
         
      return last_inc_i;
   }
   
   void operator++()
   {
      int iter_aut=vec_it.size()-1;
      incr_iter(iter_aut);
   }
      
   private:
   vector<T> vec_it;
   vector<T> vec_it_begin;
   vector<T> vec_it_end;
};

clock_ref_set get_free_vars_in_relation(const convex_clock_val_set& c);
clock_ref_set get_constant_vars_in_relation(const convex_clock_val_set& c);

#endif
