#include <iostream>
#include <list>
#include <limits>
#include <boost/dynamic_bitset.hpp>
#include "IterationVectorsMDP.h"

using namespace std;
using namespace boost;

namespace prohver {
  IterationVectorsMDP::IterationVectorsMDP(SparseNonDet &__mc) :
    mc(__mc) {
    stopCriterion = relative;
    precision = 1.0E-8;
    minProb = false;
  }

  void IterationVectorsMDP::setMinProb(const bool __minProb) {
    minProb = __minProb;
  }

  bool IterationVectorsMDP::unboundedPreciseEnough() {
    for (unsigned state = 0; state < unifProbs.size(); state++) {
      if (relative == stopCriterion) {
	if (fabs(unifProbsPrimed[state] - unifProbs[state])
	    / unifProbsPrimed[state] > precision) {
	  return false;
	}
      } else if (absolute == stopCriterion) {
	if (std::abs(unifProbs[state] - unifProbsPrimed[state])
	    >= precision) {
	  return false;
	}
      } else {
	assert(false);
      }
    }
    
    return true;
  }


  void IterationVectorsMDP::matrixVectorMultiply
  (SparseNonDet &mc, const std::vector<double> &unifProbs,
   std::vector<double> &unifProbsPrimed) {
    for (unsigned state(0); state < mc.getNumStates(); state++) {
      unifProbsPrimed[state] = minProb ? 1.0 : 0.0;
      const unsigned stateBegin(mc.getStateBegin(state));
      const unsigned stateEnd(mc.getStateEnd(state));
      for (unsigned choice(stateBegin); choice < stateEnd; choice++) {
	double prob = 0.0;
	const unsigned choiceBegin(mc.getChoiceBegin(choice));
	const unsigned choiceEnd(mc.getChoiceEnd(choice));
	for (unsigned succNr(choiceBegin); succNr < choiceEnd; succNr++) {
	  const unsigned succState(mc.cols[succNr]);
	  const double succProb(mc.nonZeros[succNr]);
	  prob += succProb * unifProbs[succState];
	}
	unifProbsPrimed[state] = minProb
	  ? std::min(unifProbsPrimed[state], prob)
	  : std::max(unifProbsPrimed[state], prob);
      }
      if (targets[state]) {
	unifProbsPrimed[state] = 1.0;
      } else {
	if (stateBegin == stateEnd) {
	  unifProbsPrimed[state] = 0.0;
	}
      }
    }
  }

  void IterationVectorsMDP::matrixVectorMultiply() {
    matrixVectorMultiply(mc, unifProbs, unifProbsPrimed);
  }

  const SparseNonDet &IterationVectorsMDP::getModel() const {
    return mc;
  }

  SparseNonDet &IterationVectorsMDP::getModel() {
    return mc;
  }

  void IterationVectorsMDP::setStopCriterion
  (const StopCriterion stopCriterion__) {
    stopCriterion = stopCriterion__;
  }
  
  void IterationVectorsMDP::setPrecision(const double precision__) {
    precision = precision__;
  }


  /**
   * Resize to size neccessary for a number of @a newSize states.
   *
   * @param newSize number of states to resize for
   */
  void IterationVectorsMDP::resize(const unsigned newSize) {
    targets.resize(newSize);
    unifProbs.resize(newSize);
    unifProbsPrimed.resize(newSize);
    result.resize(newSize);
  }

  /**
   * Perform backward iteration. This method can be used by the bounded until,
   * the instantaneous rewards and other algorithms. For bounded until, the
   * unifProbs vector should be resized and set to 1 for target states and 0
   * for the other ones. For instantaneous rewards, unifProbs should contain
   * the respective state rewards. Results will be placed in results.
   *
   * @param time time bound
   */
  void IterationVectorsMDP::backwardIterate
  (const double timeDouble) {
    const unsigned time = unsigned(timeDouble);

    unifProbsPrimed.resize(unifProbs.size());
    result.resize(unifProbs.size());
    for (unsigned state = 0; state < unifProbs.size(); state++) {
      unifProbsPrimed[state] = 0.0;
    }

    for (unsigned step = 1; step <= time; step++) {
      matrixVectorMultiply();
      unifProbs.swap(unifProbsPrimed);
    }

    result = unifProbs;
  }

  void IterationVectorsMDP::backwardIterate() {
    unifProbsPrimed.resize(unifProbs.size());
    result.resize(unifProbs.size());
    for (unsigned state = 0; state < unifProbs.size(); state++) {
      unifProbsPrimed[state] = 0.0;
    }

    unsigned iterations = 0;
    while (!unboundedPreciseEnough()) {
      iterations++;
      matrixVectorMultiply();
      unifProbs.swap(unifProbsPrimed);
    }

    result = unifProbs;
    computeScheduler();
  }

  void IterationVectorsMDP::unboundedReachBackwards() {
    unifProbs.resize(mc.getNumStates());
    unifProbsPrimed.resize(mc.getNumStates());
    for (unsigned state = 0; state < unifProbs.size(); state++) {
      unifProbs[state] = (targets[state] ? 1.0 : 0.0);
      unifProbsPrimed[state] = (targets[state] ? 1.0 : 0.0);
    }
    
    backwardIterate();
  }

  void IterationVectorsMDP::boundedReachBackwardsBounded
  (const double time) {
    unifProbs.resize(mc.getNumStates());
    for (unsigned state = 0; state < unifProbs.size(); state++) {
      unifProbs[state] = (targets[state] ? 1.0 : 0.0);
      unifProbsPrimed[state] = (targets[state] ? 1.0 : 0.0);
    }
    
    backwardIterate(time);    
  }

  /**
   * Compute scheduler from unbounded reachability probabilities.
   * Algorithm taken from orange model checking book.
   */
  void IterationVectorsMDP::computeScheduler() {
    scheduler.assign(mc.getNumStates(), numeric_limits<unsigned>::max());
    for (unsigned state(0); state < mc.getNumStates(); state++) {
      const unsigned stateBegin(mc.getStateBegin(state));
      scheduler[state] = stateBegin;
    }

    dynamic_bitset<> maxActions
      (mc.getStateEnd(mc.getNumStates() - 1) , false);

    /* compute maximizing actions */
    for (unsigned state(0); state < mc.getNumStates(); state++) {
      const unsigned stateBegin(mc.getStateBegin(state));
      const unsigned stateEnd(mc.getStateEnd(state));
      for (unsigned choice(stateBegin); choice < stateEnd; choice++) {
	double prob = 0.0;
	const unsigned choiceBegin(mc.getChoiceBegin(choice));
	const unsigned choiceEnd(mc.getChoiceEnd(choice));
	for (unsigned succNr(choiceBegin); succNr < choiceEnd; succNr++) {
	  const unsigned succState(mc.cols[succNr]);
	  const double succProb(mc.nonZeros[succNr]);
	  prob += succProb * result[succState];
	}
	if (abs(prob - result[state]) < 1E-4) {
	  maxActions[choice] = true;
	}
      }
    }

    /* compute scheduler */
    /* TODO for large matrices, it would be more efficient to inverse
     * the matrix */
    dynamic_bitset<> included(mc.getNumStates(), false);

    for (unsigned state(0); state < mc.getNumStates(); state++) {
      if (targets[state]) {
	included[state] = true;
      }
    }

    bool changed;
    do {
      changed = false;
      for (unsigned state(0); state < mc.getNumStates(); state++) {
	if (!included[state]) {
	  const unsigned stateBegin(mc.getStateBegin(state));
	  const unsigned stateEnd(mc.getStateEnd(state));
	  for (unsigned choice(stateBegin); choice < stateEnd; choice++) {
	    if (maxActions[choice]) {
	      const unsigned choiceBegin(mc.getChoiceBegin(choice));
	      const unsigned choiceEnd(mc.getChoiceEnd(choice));
	      for (unsigned succNr(choiceBegin); succNr < choiceEnd; succNr++) {
		const unsigned succState(mc.cols[succNr]);
		if (included[succState]) {
		  included[state] = true;
		  changed = true;
		  scheduler[state] = choice;
		  break;
		}
	      }
	      if (included[state]) {
		break;
	      }
	    }
	  }
	}
      }
    } while (changed);
  }
}
